package org.eaglei.repository.model;

import java.util.ArrayList;
import java.util.List;

import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.OpenRDFException;
import org.openrdf.model.impl.ContextStatementImpl;
import org.openrdf.model.URI;
import org.openrdf.model.Literal;
import org.openrdf.model.Statement;
import org.openrdf.model.Value;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.model.vocabulary.RDFS;
import org.openrdf.query.BindingSet;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.TupleQueryResult;
import org.openrdf.query.TupleQueryResultHandlerBase;
import org.openrdf.query.TupleQueryResultHandlerException;
import org.openrdf.query.QueryLanguage;

import org.eaglei.repository.auth.Authentication;
import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.status.InternalServerErrorException;
import org.eaglei.repository.status.ForbiddenException;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.util.WithRepositoryConnection;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.util.Utils;

/**
 * Record class describing one "grant" of access to an entity in the repository.
 * Also includes utility methods to manage grants.
 *
 * Started April, 2010
 *
 * @author Larry Stone
 * @version $Id: $
 */
public final class AccessGrant
{
    private static final Logger log = LogManager.getLogger(Access.class);

    // SPARQL query to find all access control predicates - only need to
    // run this once but it's good to pick up on any repo ontology changes.
    private static final String ACCESS_PREDICATE_QUERY =
        "SELECT * WHERE { ?accessPred <"+RDFS.SUBPROPERTYOF+"> <"+REPO.HAS_ANY_ACCESS+">}";

    // Cached query to get imported grants, created with all current
    // access predicates.  Can't just ues a triple pattern because import
    // is in a separate memory repository from teh repo ontology.
    private static String importGrantQuery = null;

    /**
     * SPARQL query to find grants on any subject.  Need to bind ?instance
     * to the subject URI.  Output columns are:
     *  ?access - type of access being granted
     *  ?agent - grantee
     *  ?agentType - either :Role or :Person
     *  (other result columsn are optional labels for these 3 URIs)
     */
    private static final String getGrantsQueryPrefix =
        "SELECT DISTINCT * WHERE { \n"+
        "GRAPH <"+REPO.NG_INTERNAL+"> { ?instance ?access ?agent } . \n"+
        "?access <"+RDFS.SUBPROPERTYOF+"> <"+REPO.HAS_ANY_ACCESS+"> \n"+
        "OPTIONAL { ?access <"+RDFS.LABEL+"> ?accessLabel }\n"+
        "OPTIONAL { ?agent <"+RDFS.LABEL+"> ?agentLabel }\n"+
    // XXX FIXME TODO: probably don't need the UNION ...subclassOf part of
    //      this query, but try deleting it when we have some tests.
        "OPTIONAL { {{?agent <"+RDF.TYPE+"> ?agentType} UNION {?agent <"+RDFS.SUBCLASSOF+"> ?agentType}}\n"+
        "  FILTER (?agentType = <"+REPO.ROLE+"> || ?agentType = <"+REPO.PERSON+">) \n"+
        "  OPTIONAL { ?agentType <"+RDFS.LABEL+"> ?agentTypeLabel }}\n";

    private static final String getGrantsQuery =
        getGrantsQueryPrefix + "}";

    // This query matches grants to a specific user or its roles.
    // Also need to bind ?user when using this
    private static final String getMyGrantsQuery =
        getGrantsQueryPrefix +
        "{{?user <"+REPO.HAS_ROLE+"> ?agent} UNION {?instance ?access ?user}}}";

    /** record class for RDF "term" that has URI and label */
    private static final class Term
    {
        private URI uri = null;
        private String label = null;

        private Term(URI uri, String label)
        {
            Term.this.uri = uri;
            Term.this.label = (label == null) ? uri.getLocalName() : label;
        }
        private Term(URI uri)
        {
            Term.this.uri = uri;
            Term.this.label = uri.getLocalName();
        }

        /** String rendition is the URI */
        @Override
        public String toString()
        {
            return uri.toString();
        }
    }

    private Term access = null;
    private Term agent = null;
    private Term agentType = null;

    /**
     * <p>Constructor for AccessGrant.</p>
     *
     * @param agent a {@link org.openrdf.model.URI} object.
     * @param agentLabel a {@link java.lang.String} object.
     * @param agentType a {@link org.openrdf.model.URI} object.
     * @param agentTypeLabel a {@link java.lang.String} object.
     * @param access a {@link org.openrdf.model.URI} object.
     * @param accessLabel a {@link java.lang.String} object.
     */
    private AccessGrant(URI agent, String agentLabel, URI agentType, String agentTypeLabel, URI access, String accessLabel)
    {
        AccessGrant.this.agent = new Term(agent, agentLabel);
        AccessGrant.this.agentType = new Term(agentType, agentTypeLabel);
        AccessGrant.this.access = new Term(access, accessLabel);
    }

    /** consolidated getter */
    public URI getAccessURI()
    {
        return access == null ? null : access.uri;
    }
    /** consolidated getter */
    public String getAccessLabel()
    {
        return access == null ? null : access.label;
    }
    /** consolidated getter */
    public URI getAgentURI()
    {
        return agent == null ? null : agent.uri;
    }
    /** consolidated getter */
    public String getAgentLabel()
    {
        return agent == null ? null : agent.label;
    }
    /** consolidated getter */
    public URI getAgentTypeURI()
    {
        return agentType == null ? null : agentType.uri;
    }
    /** consolidated getter */
    public String getAgentTypeLabel()
    {
        return agentType == null ? null : agentType.label;
    }


    /**
     * Remove specified grant of access from an instance.
     * Returns true if grant was there, false if not.
     * WARNING: You will need to commit() these changes to the repo connection!
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param instance subject from which to remove access grant a {@link org.openrdf.model.URI} object.
     * @param agent principal to whom the access was granted, a {@link org.openrdf.model.URI} object.
     * @param access type of access, a {@link org.openrdf.model.URI} object.
     * @return a boolean, true if there was a grant to be removed.
     */
    public static boolean removeGrant(HttpServletRequest request, URI instance, URI agent, URI access)
    {
        if (Access.hasPermission(request, instance, Access.ADMIN)) {
            return removeGrantAsAdministrator(request, instance, agent, access);
        } else
              throw new ForbiddenException("You are not allowed to change access controls on "+instance);
    }

    /**
     * Remove specified grant of access from a URI, but WITHOUT cehcking for
     * ADMIN access.  This is meant for INTERNAL user where the program
     * logic mediates access, e.g. workflow.
     * Returns true if grant was there, false if not.
     * WARNING: You will need to commit() these changes to the repo connection!
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param instance subject from which to remove access grant a {@link org.openrdf.model.URI} object.
     * @param agent principal to whom the access was granted, a {@link org.openrdf.model.URI} object.
     * @param access type of access, a {@link org.openrdf.model.URI} object.
     * @return a boolean, true if there was a grant to be removed.
     */
    public static boolean removeGrantAsAdministrator(HttpServletRequest request, URI instance, URI agent, URI access)
    {
        // sanity check: do not allow null, which is wildcard delete:
        if (instance == null || access == null || agent == null)
            throw new BadRequestException("removeGrant called with an illegal null URI.");
        if (Access.isAccessPredicate(access)) {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            try {
                if (rc.hasStatement(instance, access, agent, false, REPO.NG_INTERNAL)) {
                    rc.remove(instance, access, agent, REPO.NG_INTERNAL);
                    return true;
                } else {
                    return false;
                }
            } catch (OpenRDFException e) {
                log.error(e);
                throw new InternalServerErrorException("Failed in remove ACL: ",e);
            }
        } else {
            throw new IllegalArgumentException("Access URI is not a valid access predicate: "+access.stringValue());
        }
    }

    /**
     * Add the specified grant to the instance.  Requires ADMIN access.
     * WARNING: You will need to commit() these changes to the repo connection!
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param instance subject from which to add access grant a {@link org.openrdf.model.URI} object.
     * @param agent principal to whom the access was granted, a {@link org.openrdf.model.URI} object.
     * @param access type of access, a {@link org.openrdf.model.URI} object.
     */
    public static void addGrant(HttpServletRequest request, URI instance, URI agent, URI access)
    {
        if (Access.hasPermission(request, instance, Access.ADMIN)) {
            addGrantAsAdministrator(request, instance, agent, access);
        } else
            throw new ForbiddenException("You are not allowed to change access controls on "+instance);
    }

    /**
     * Add the specified grant to the instance, but WITHOUT cehcking for
     * ADMIN access.  This is meant for INTERNAL user where the program
     * logic mediates access, e.g. workflow.
     * WARNING: You will need to commit() these changes to the repo connection!
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param instance subject from which to add access grant a {@link org.openrdf.model.URI} object.
     * @param agent principal to whom the access was granted, a {@link org.openrdf.model.URI} object.
     * @param access type of access, a {@link org.openrdf.model.URI} object.
     */
    public static void addGrantAsAdministrator(HttpServletRequest request, URI instance, URI agent, URI access)
    {
        if (Access.isAccessPredicate(access)) {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            try {
                if (!rc.hasStatement(instance, access, agent, false, REPO.NG_INTERNAL)) {
                    rc.add(instance, access, agent, REPO.NG_INTERNAL);
                }
            } catch (OpenRDFException e) {
                log.error(e);
                throw new InternalServerErrorException("Failed in add ACL: ",e);
            }
        } else {
            throw new IllegalArgumentException("Access URI is not a valid access predicate: "+access.stringValue());
        }
    }

    /**
     * Find all grants on the given subject URI
     * @param request
     * @param uri subject of grants
     * @return Iterable of grants on the given subject
     */
    public static Iterable<AccessGrant> getGrants(HttpServletRequest request, URI uri)
    {
        return getGrantsInternal(request, WithRepositoryConnection.get(request), uri, false);
    }

    /**
     * Find all grants on the given subject URI pertaining to current authenticated
     * user.
     * @param request
     * @param uri subject of grants
     * @return Iterable of grants on the given subject
     */
    public static Iterable<AccessGrant> getMyGrants(HttpServletRequest request, URI uri)
    {
        return getGrantsInternal(request, WithRepositoryConnection.get(request), uri, true);
    }

    /**
     * Get list of access grants on this instance
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param rc the repository connection - MIGHT not be from request
     * @param uri subject on which to find grants, a {@link org.openrdf.model.URI} object.
     * @param mine when true, only return grants affecting current user
     * @return all grants in a {@link java.lang.Iterable} object, possibly empty.
     */
    private static Iterable<AccessGrant> getGrantsInternal(HttpServletRequest request,
                                          RepositoryConnection rc,
                                          URI uri, boolean mine)
    {
        try {
            String qs = mine ? getMyGrantsQuery : getGrantsQuery;
            TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, qs);
            q.setBinding("instance", uri);
            if (mine) {
                if (request == null)
                    throw new IllegalArgumentException("Cannot get grants for current user when request is null.");
                q.setBinding("user", Authentication.getPrincipalURI(request));
            }
            q.setDataset(Access.ACCESS_DATASET);
            grantHandler h = new grantHandler();
            SPARQL.evaluateTupleQuery(qs, q, h);
            return h.result;
        } catch (OpenRDFException e) {
            log.error(e);
            throw new InternalServerErrorException("Failed in query: "+e,e);
        }
    }

    /**
     *  Translate internal grant objects into exportable statements on a
     *  given subject.
     *
     * @param uri - the subject
     * @param grants - grant objects
     * @return iterable list of Sesame Statement objects
     */
    public static Iterable<Statement> exportGrants(URI uri, Iterable<AccessGrant> grants)
    {
        List<Statement> result = new ArrayList<Statement>();
        for (AccessGrant g : grants) {
            result.add(new ContextStatementImpl(uri, g.access.uri, g.agent.uri, REPO.NG_INTERNAL));
        }
        return result;
    }

    /**
     * Get importable access grant statements for URI from import document.
     * NOTE that it looks up grants on oldURI, but puts newURI in the
     * new grant statements; this is in case subject URI got transformed
     * on the import.
     * Also note that source grants come from 'content' (could be a memory repo)
     * but we need still the regular repo to compute ACL query the first time.
     * @param request
     * @param content
     * @param oldURI
     * @param newURI
     * @return Iterable of grant RDF Statemens to be imported
     */
    public static Iterable<Statement> importGrants(HttpServletRequest request,
                                        RepositoryConnection content, URI oldURI, URI newURI)
    {
        String qs = getImportGrantQuery(request);
        try {
            TupleQuery q = content.prepareTupleQuery(QueryLanguage.SPARQL, qs);
            q.setDataset(Access.ACCESS_DATASET);
            q.setBinding("s", oldURI);
            TupleQueryResult qr = null;
            List<Statement> result = new ArrayList<Statement>();
            try {
                qr = SPARQL.evaluateTupleQuery(qs, q);
                while (qr.hasNext()) {
                    BindingSet bs= qr.next();
                    result.add(new ContextStatementImpl(newURI, (URI)bs.getValue("p"),
                           bs.getValue("o"), REPO.NG_INTERNAL));
                }
            } finally {
                if (qr != null)
                    qr.close();
            }
            log.debug("importGrants("+oldURI.stringValue()+"): returning "+result.size()+" grant statements.");
            return result;
        } catch (OpenRDFException e) {
            log.error("Failed in one of the Import Grant queries: ",e);
            throw new InternalServerErrorException(e);
        }
    }

    // compute the grant query based on grant predicates in current onto:
    private static String getImportGrantQuery(HttpServletRequest request)
    {
        try {
            if (importGrantQuery == null) {
                RepositoryConnection rc = WithRepositoryConnection.get(request);
                TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, ACCESS_PREDICATE_QUERY);
                q.setDataset(Access.ACCESS_DATASET);
                TupleQueryResult qr = null;
                StringBuilder igq = new StringBuilder("SELECT ?p ?o WHERE {?s ?p ?o FILTER(" /*)}*/);
                try {
                    qr = SPARQL.evaluateTupleQuery(ACCESS_PREDICATE_QUERY, q);
                    boolean first = true;
                    while (qr.hasNext()) {
                        String ap = Utils.valueAsString(qr.next().getValue("accessPred"));
                        if (first)
                            first = false;
                        else
                            igq.append(" || ");
                        igq.append("?p = <").append(ap).append(">");
                    }
                    igq.append(/*{(*/ ")}");
                    importGrantQuery = igq.toString();
                    log.debug("Generated fixed Access Import query: \""+importGrantQuery+"\"");
                } finally {
                    if (qr != null)
                        qr.close();
                }
            }
            return importGrantQuery;
        } catch (OpenRDFException e) {
            log.error("Failed in one of the Import Grant queries: ",e);
            throw new InternalServerErrorException(e);
        }
    }


    // Grant query result handler: stash tuple query results in List<Grant>
    // Looks for bindings of "uri" and "label"
    private static class grantHandler extends TupleQueryResultHandlerBase
    {
        private List<AccessGrant> result = new ArrayList<AccessGrant>();

        protected grantHandler()
        {
            super();
        }

        // columns: namedGraphURI, namedGraphLabel, typeURI, typeLabel, anon
        @Override
        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            Value agent = bs.getValue("agent");
            Value agentLabel = bs.getValue("agentLabel");
            Value agentType = bs.getValue("agentType");
            Value agentTypeLabel = bs.getValue("agentTypeLabel");
            Value access = bs.getValue("access");
            Value accessLabel = bs.getValue("accessLabel");

            if (!(agent instanceof URI))
                throw new TupleQueryResultHandlerException(
                    "The value for 'agent' was null or not a URI type in grantHandler: "+agent);
            else if (!(access instanceof URI))
                throw new TupleQueryResultHandlerException(
                    "The value for 'access' was null or not a URI type in grantHandler: "+access);
            else {
                // kludge: if agentType not set, grab repo:Agent
                if (agentType == null)
                    agentType = REPO.AGENT;
                // is this grant from the repo ontology, hence immutable??
                if (log.isDebugEnabled())
                    log.debug("getGrants: Adding Grant(agent="+agent+
                        ", agentLabel="+agentLabel+", agentType="+agentType+
                        ", agentTypeLabel="+agentTypeLabel+
                        ", access="+access+", accessLabel="+accessLabel);
                result.add(new AccessGrant((URI)agent,
                    (agentLabel instanceof Literal) ? ((Literal)agentLabel).getLabel() : null,
                    (URI)agentType,
                    (agentTypeLabel instanceof Literal) ? ((Literal)agentTypeLabel).getLabel() : null,
                    (URI)access,
                    (accessLabel instanceof Literal) ? ((Literal)accessLabel).getLabel() : null));
            }
        }
    }
}
