package org.eaglei.search.common;

import org.eaglei.search.request.SearchRequest;
import org.eaglei.search.request.SearchResultSet;

/**
 * 
 * @author Clint Gilbert
 * 
 * Feb 9, 2010
 *
 * Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 *
 * @param <T>
 */
public interface Serializer<T> {
    
    String serialize(final T object) throws SerializationException;
    
    T deserialize(final String serialized) throws SerializationException;
    
    public static final Serializer<SearchRequest> SearchRequestSerializer = new Serializer<SearchRequest>() {
        @Override
        public String serialize(final SearchRequest request) throws SerializationException {
            assert request != null;
            
            //TODO: How are errors handled? returning null?
            return request.toURLParams();
        }
        
        @Override
        public SearchRequest deserialize(final String serialized) throws SerializationException {
            assert serialized != null;
            
            final SearchRequest request = SearchRequest.parse(serialized);
            
            //TODO: How are errors handled? Is null an error?
            if(request == null) {
                throw new SerializationException("Error unmarshalling SearchRequest; got null result for: '" + serialized + "'");
            }
            
            return request;
        }
    };
    
    public static final Serializer<SearchResultSet> SearchResultSetSerializer = new Serializer<SearchResultSet>() {
        @Override
        public String serialize(final SearchResultSet resultSet) throws SerializationException {
            return GsonHolder.Gson.toJson(resultSet, SearchResultSet.class);
        }
        
        @Override
        public SearchResultSet deserialize(final String serialized) throws SerializationException {
            return GsonHolder.Gson.fromJson(serialized, SearchResultSet.class);
        }
    };
    
    public static final Serializer<String> NullSerializer = new Serializer<String>() {
        @Override
        public String deserialize(final String serialized) throws SerializationException {
            return serialized;
        }

        @Override
        public String serialize(final String object) throws SerializationException {
            return object;
        }
    };
}
