package org.eaglei.search.provider;

import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.model.EIEntity;
import org.eaglei.search.request.SearchRequest;
import org.eaglei.search.request.SearchResult;
import org.eaglei.search.request.SearchResultSet;

/**
 * Trivial implementation of SearchResultMerger that uses result rank. Also trims the results sets
 * to the specified range. 
 * NOTE: the rank across result sets is not in most cases comparable.
 * 
 * @author rfrost
 */
public class SearchResultRankMerger implements SearchProvider {

    private static final Log logger = LogFactory.getLog(SearchResultRankMerger.class);
    private static final boolean DEBUG = logger.isDebugEnabled();
    private MultiNodeSearchProvider nestedProvider;
    private boolean trimToRange = true;
    
    public SearchResultRankMerger(final MultiNodeSearchProvider nestedProvider) {
        this.nestedProvider = nestedProvider;
    }

    /**
     * Sets the flag that controls whether the returned SearchResultSet is trimmed to the range specified in the SearchRequest
     * or whether all merged results are returned. In both cases, the SearchRequest range is used when calling the nested SearchProvider.
     * @param trimToRange True to trim merged results.
     */
    public void setTrimToRange(final boolean trimToRange) {
       this.trimToRange = trimToRange; 
    }
    
    /**
     * @see #setTrimToRange(boolean)
     */
    public boolean getTrimToRange() {
        return this.trimToRange;
    }
    
    /* (non-Javadoc)
     * @see org.eaglei.search.provider.SearchProvider#init()
     */
    public void init() throws IOException {
        this.nestedProvider.init();
    }
    
    /* (non-Javadoc)
     * @see org.eaglei.search.provider.SearchProvider#getInstitutions()
     */
    public Collection<EIEntity> getInstitutions() {
        return this.nestedProvider.getInstitutions();
    }
    
    /* (non-Javadoc)
     * @see org.eaglei.search.provider.SearchProvider#query(org.eaglei.search.request.SearchRequest)
     */
    public SearchResultSet query(final SearchRequest request) throws IOException {
        // always request the result set starting from the beginning with max adjusted by the original
        // start range, we will trim the beginning for pagination in the merge call
        final int startIndex = request.getStartIndex();
        request.setStartIndex(0); 
        request.setMaxResults(startIndex + request.getMaxResults()); 
        
        // call the nexted provider
        final Collection<SearchResultSet> results = nestedProvider.query(request);
        
        // reset the start index and merge the results into a single SearchResultSet based on rank
        request.setStartIndex(startIndex);
        return merge(results, request);
    }
    
    private SearchResultSet merge(final Collection<SearchResultSet> results, final SearchRequest request) {
	    SearchResultSet merged = null;
	    if (results.size() == 1) {
			// if there is only one, no need to merge
            if (DEBUG) {	        
                logger.debug("Collection only included one SearchResultSet, returning that");
            }
			merged = results.iterator().next();
		} else if (results.size() == 0) {
			// if empty, create an empty result set
            if (DEBUG) {
		        logger.debug("Collection was empty returning empty SearchResultSet");
		    }
			final SearchResultSet empty = new SearchResultSet(request);
			empty.setTotalCount(0);
			return empty;
		} else {
			// merge, ordered by rank
			merged = new SearchResultSet(request);
			final List<SearchResult> mergedResults = merged.getResults();
			final SortedSet<SearchResult> sortedResults = new TreeSet<SearchResult>();
            if (DEBUG) {
                logger.debug("Merging " + results.size() + " SearchResultSets");                
			}
            int start = 0;
			for (SearchResultSet resultSet: results) {
                if (DEBUG) {
                    logger.debug("Merging result set with " + resultSet.getResults().size() + " results and start " + resultSet.getStartIndex());
	            }
				merged.setTotalCount(merged.getTotalCount() + resultSet.getTotalCount());
				for (SearchResult result: resultSet.getResults()) {
				    if (!sortedResults.contains(result)) {
	                    sortedResults.add(result);
				    } else {
				        // decrement the total count
				        merged.setTotalCount(merged.getTotalCount()-1);
				    }
				}
			}
			for (SearchResult result: sortedResults) {
				merged.getResults().add(result);
			}
            if (DEBUG) {
                logger.debug("Size of merged result set: " + merged.getResults().size());
            }
		}
	    if (getTrimToRange()) {
	        return trimToRange(merged, request);
	    }
	    return merged;
	}
	
	// instead of trimming these result sets, want to instead be caching the full set and returning copies
	// of the page ranges
	private SearchResultSet trimToRange(final SearchResultSet results, final SearchRequest request) {
        if (DEBUG) {
	        logger.debug("Trimming to range " + request.getStartIndex() + " to " + (request.getStartIndex() + 
	                request.getMaxResults()));
	    }
	    final SearchResultSet trimmed = new SearchResultSet(request);
	    trimmed.setStartIndex(request.getStartIndex());
	    trimmed.setTotalCount(results.getTotalCount());
	    final List<SearchResult> list = results.getResults();

	    // filter out any portions of the results before the requested start index
	    int numToRemoveOnFront = request.getStartIndex() - results.getStartIndex();
	    if (numToRemoveOnFront < 0) {
	        numToRemoveOnFront = 0;
	    }
	    for (int i = numToRemoveOnFront; i < (numToRemoveOnFront+ request.getMaxResults()) && i < list.size(); i++) {
	        trimmed.getResults().add(list.get(i));
	    }
	       if (DEBUG) {
	            logger.debug("Trimmed result start: " + trimmed.getStartIndex() + ", trimmed size " + trimmed.getResults().size());
	        }
	    return trimmed;
	}

}
