package org.eaglei.search.logging;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLClassLoader;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.text.SimpleDateFormat;

import java.util.Enumeration;
import java.util.Properties;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.search.provider.SearchRequest;

import com.mysql.jdbc.Driver;

/**
 * Class providing asynchronous logging of events using
 * a BlockingQueue.
 * 
 * Specify a new or existing table in the constructor, along
 * with the build ID and ontology ID. If you do not want
 * one or the other of the IDs, set them to an empty string
 * or to null.
 * 
 * Extend this abstract class to create a new logger for a
 * specified table. Implement the abstract methods with the
 * details of the table and rows. See AsynchronousLogger for an example.
 * 
 * @author RTS6
 *
 */
public abstract class AsynchronousLogger {

	protected static final Log logger = LogFactory.getLog(AsynchronousLogger.class);	    

	// DB connection
	protected Connection conn = null;
	// Time between DB connection retries, if connection is lost while logging.
	private final int RETRY_INTERVAL = 30000;
	
	// Logging queue, drained by LoggerDaemon
	private final BlockingQueue<LogInfo> logQueue = new LinkedBlockingQueue<LogInfo>(); 
	
	
	protected SimpleDateFormat dateFormat = new SimpleDateFormat("MMM dd, yyyy HH:mm:ss.SSS");
	
	protected String buildID = "";
	protected String ontologyID = "";	
	protected String tableName = null;
	
	private String loggerName = "";
	
	protected PreparedStatement insertCmd = null;
	
	public AsynchronousLogger(String tname, String buildVersion, String ontVersion) {
		tableName = tname;
		buildID = buildVersion;
		ontologyID = ontVersion;
		init();
	}
	
	// Default connection info
	private static final String JDBC = "jdbc:mysql://";
	private static final String LOCALHOST = "localhost:3306";
	private static final String DB_NAME = "searchlogdb";
	private static final String DB_USER = "searchlog";
	private static final String DB_PASSWORD = "s34rchl0g";
	
	// Live connection info
	private String jdbc = JDBC;
	private String localhost = LOCALHOST;
	private String db_name = DB_NAME;
	private String db_user = DB_USER;
	private String db_password = DB_PASSWORD;
	
	private static final String UNK = "Unknown";

	/*
	 * Connect to DB. Create tables, if needed. Start logging thread.
	 */
	private void init() {
		
		checkBuildOntID();
		
		getDBproperties();
		
		connectDB(false);		
		if (conn == null) {
			return;
		}
		
		createTable();
		if (conn == null) {
			return;
		}
		
		setInsertCmd();
		
		startLogging();
         
        logger.info("Asynchronous Logger started for table " + tableName);
	}
	
	private void checkBuildOntID() {
		if (ontologyID == null) {
			ontologyID = UNK;
		} else if (ontologyID.trim().length() == 0) {
			ontologyID = UNK;
		}
		if (buildID == null) {
			buildID = UNK;
		} else if (buildID.trim().length() == 0) {
			buildID = UNK;
		}
		logger.info("AsynchronousLogger using " + buildID + " as build identifier, and " + ontologyID + " as ontology identifer.");
	}
	
	// Get DB connection info
	private void getDBproperties() {
		InputStream stream = this.getClass().getClassLoader().getResourceAsStream("asynclogger.properties");
		if (stream == null) {
			logger.warn("AsynchronousLogger could not find asynclogger.properties. Using default values.");
			return;
		}
		Properties props = new Properties();
		try {
			props.load(stream);
		} catch (IOException e) {
			logger.warn("AsynchronousLogger found but could not read asynclogger.properties. Using default values.");
			return;
		}
		jdbc = props.getProperty("JDBC", JDBC);
		localhost = props.getProperty("LOCALHOST", LOCALHOST);
		db_name = props.getProperty("DB_NAME", DB_NAME);
		db_user = props.getProperty("DB_USER", DB_USER);
		db_password = props.getProperty("DB_PASSWORD", DB_PASSWORD);
	}
	
	// Create DB connection
	private void connectDB(boolean silent) {
		
		String url = jdbc + localhost + "/" + db_name;
		
		if (!silent) {
			// DEBUG
			String driverName = "org.gjt.mm.mysql.Driver";
			//String driverName = "xyzzy";
			try {
				Class.forName(driverName);
			} catch (ClassNotFoundException e) {
				logger.error("AsynchronousLogger could not find JDBC driver " + driverName);				
				
				logger.info("Thread name: " + Thread.currentThread().getName());
				
				Enumeration<java.sql.Driver> drivers = DriverManager.getDrivers();
				if (drivers.hasMoreElements()) {
					logger.info("Enumerating known drivers:");
					while (drivers.hasMoreElements()) {
						java.sql.Driver d = drivers.nextElement();
						logger.info("    " + d.getClass().getName());
					}
				} else {
					logger.info("There are no known drivers.");
				}
				
				ClassLoader cloader = this.getClass().getClassLoader();
				StringBuffer classpath = new StringBuffer();
				URL[] urls = ((URLClassLoader)cloader).getURLs();
				for(int i=0; i < urls.length; i++) {
			        classpath.append(urls[i].getFile()).append("\r\n");
			     }    
		         logger.info("Classpath: " + classpath.toString());
		      
		         logger.info("Will attempt to connect with the following URL and credentials: " + url
		    		  + "   User: " + db_user + "   Password: " + db_password);
			}
			// DEBUG
		}
		
		try {
			conn = DriverManager.getConnection(url, db_user, db_password);
		} catch (Exception e) {
			if (!silent) {
				logger.warn("AsynchronousLogger failed to get a database connection.", e);
			}
//			e.printStackTrace();
			return;
		}
	}
	
	// Create table for log entries, if it does not exist.
	// See AsychronousLoggerSearch for an example.
	protected abstract void createTable();
	
	// Create a PreparedStatement to insert a row into the table.
	protected abstract void setInsertCmd();
	
	// Fill the insert command with data for a new row.
	protected abstract void fillInInsertCmd(LogInfo entry);
	
	// Start logging thread.
	private void startLogging() {
		loggerName = "AsyncLogger_" + tableName;
		Thread logThread = new Thread(new LoggerDaemon(), loggerName);
        logThread.setPriority(Thread.MIN_PRIORITY);
        logThread.setDaemon(true);
        logThread.start();
	}
	
	public void log(LogInfo entry) {
		if (conn == null) {
			return;
		}
		logQueue.add(entry);
	}
	
	public Connection getConnection() {
		return conn;
	}
	
	public String getBuildID() {
		return buildID;
	}
	
	public String getOntologyID() {
		return ontologyID;
	}
	
	public void finalize() {
		try {
			if (conn != null) {
				conn.close();
				conn = null;
			}
		} catch(SQLException e) {
			e.printStackTrace();
		}	
	}
	
	//===================
	// Runnable logger
	private class LoggerDaemon implements Runnable {
	
		@Override
		public void run() {
			
			try {
				while(true && (conn!= null)) {
					LogInfo entry = logQueue.take();
					writeDB(entry);
				}
			} catch (InterruptedException e) {
				// Ignore
			}
		}
		
		private void writeDB (LogInfo entry) {
			try {				
				fillInInsertCmd(entry);
				insertCmd.execute();
			} catch (Throwable e) {
				logger.error(loggerName + " failed to log an entry, possible communication failure. Will try to re-open connection.");
				//e.printStackTrace();
				conn = null;
				while (conn == null) {
					try {
						Thread.sleep(RETRY_INTERVAL);
					} catch (InterruptedException e1) {
						// ignore
					}
					connectDB(true);
				}
				logger.info(loggerName + " re-acquired database connection.");
				setInsertCmd(); // Re-create prepared statement for insert
			}
		}
	}
}
