package org.eaglei.search.logging;

import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.Date;
import java.util.UUID;

import org.eaglei.model.EIInstance;
import org.eaglei.model.EIURI;
import org.eaglei.search.provider.SearchRequest;
import org.eaglei.security.Session;

public class AsynchronousLoggerInstance extends AsynchronousLogger {

	public static final int ROW_UUID_COL 			= 1;
	public static final int SESSIONID_COL 			= 2;
	public static final int USERID_COL 				= 3;
	public static final int USERINSTITUTION_COL 	= 4;
	public static final int DATE_COL 				= 5;
	public static final int START_COL 				= 6;
	public static final int END_COL 				= 7;
	public static final int DURATION_COL 			= 8;
	public static final int INSTANCEID_COL 			= 9;
	public static final int INSTANCECLASS_COL 		= 10;
	public static final int INSTANCELABEL_COL 		= 11;
	public static final int INSTANCETYPE_COL 		= 12;
	public static final int BUILDID_COL 			= 13;
	public static final int ONTOLOGYID_COL 			= 14;
	
	public AsynchronousLoggerInstance(String tname, String buildVersion,
			String ontVersion) {
		super(tname, buildVersion, ontVersion);
	}

	@Override
	protected void createTable() {
		if (tableName == null) {
			logger.error("Table name was null.");
			conn = null;
			return;
		}
		if (tableName.trim().length() == 0) {
			logger.error("Table name was empty.");
			conn = null;
			return;
		}
		
		String createTableCmd = "CREATE TABLE IF NOT EXISTS " + tableName.trim() + " (" + 
			"op_id varchar(128) character set ascii NOT NULL," +  
			"session_id varchar(128) character set ascii NOT NULL," +  
			"user_id varchar(128) character set ascii NOT NULL," + 
			"user_institution varchar(128) character set ascii NOT NULL," + 
			"date varchar(64) character set ascii NOT NULL," +
			"start bigint(20) NOT NULL," + 
			"end bigint(20) NOT NULL," + 
			"duration bigint(20) NOT NULL," + 
			"instance_id varchar(128) character set ascii NOT NULL," +
			"instance_class varchar(128) character set ascii NOT NULL," +
			"instance_label varchar(128) character set ascii NOT NULL," +
			"instance_type varchar(128) character set ascii NOT NULL," +
			"build_id varchar(128) character set ascii NOT NULL," + 
			"ontology_id varchar(128) character set ascii NOT NULL" +
			") ENGINE=MyISAM DEFAULT CHARSET=utf8;";
		
		try {
			PreparedStatement createStmt = conn.prepareStatement(createTableCmd);
			createStmt.execute();
		} catch (SQLException e) {
			logger.error("Asynchronous logger failed to create table " + tableName);
			e.printStackTrace();
			try {
				conn.close();
			} catch (SQLException e1) {
				e1.printStackTrace();
			}
			conn = null;
			return;
		}
	}

	@Override
	protected void fillInInsertCmd(LogInfo entryArg) {
		
		if ( !(entryArg instanceof LogInfoInstance) ) {
			logger.error("Incorrect LogInfo type");
			return;
		}
		
		LogInfoInstance entry = (LogInfoInstance) entryArg;
		
		try {
			UUID opid = UUID.randomUUID();
			long end = entry.getEnd();
			long duration = end - entry.getStart();
			
			String date = dateFormat.format(new Date(end));
			//System.out.println("Date:  " + date);
			
			insertCmd.setString(ROW_UUID_COL, opid.toString());
			insertCmd.setString(SESSIONID_COL, entry.getSessionID());
			insertCmd.setString(USERID_COL, entry.getUserID());
			insertCmd.setString(USERINSTITUTION_COL, entry.getUserInstitution());
			insertCmd.setString(DATE_COL, date);
			insertCmd.setLong(START_COL, entry.getStart());
			insertCmd.setLong(END_COL, entry.getEnd());
			insertCmd.setLong(DURATION_COL, duration);
			insertCmd.setString(INSTANCEID_COL, entry.getInstanceID());
			insertCmd.setString(INSTANCECLASS_COL, entry.getInstanceClass());
			insertCmd.setString(INSTANCELABEL_COL, entry.getInstanceLabel());
			insertCmd.setString(INSTANCETYPE_COL, entry.getInstanceType());
			insertCmd.setString(BUILDID_COL, entry.getBuildID());
			insertCmd.setString(ONTOLOGYID_COL, entry.getOntologyID());
			
			//System.out.println("INSERT COMMAND:  " + insertCmd.toString());
		} catch (SQLException sqle) {
			logger.error("AsynchronousLogger failed to fill in insert command: " + insertCmd.toString());
		}
	}

	@Override
	protected void setInsertCmd() {
	
		if (conn != null) {
			String INSERT_CMD = "INSERT INTO " + tableName + " ( " + 
				"op_id, session_id, user_id, user_institution, date, start, end, " + 
				"duration, instance_id, instance_class, instance_label, instance_type," +
				" build_id, ontology_id ) VALUES ( " +
				"?, " + // op_id
				"?, " + // session_id
				"?, " + // user_id
				"?, " + // user_institution
				"?, " + // date
				"?, " + // start
				"?, " + // end
				"?, " + // duration
				"?, " + // instance_id
				"?, " + // instance_class
				"?, " + // instance_label
				"?, " + // instance_type
				"?, " + // build_id
				"?"     // ontology_id
				+ " );";
			try {
				insertCmd = conn.prepareStatement(INSERT_CMD);
			} catch (SQLException e) {
				logger.error("Asynchronous Logger failed to create insert command");
				e.printStackTrace();
				try {
					conn.close();
				} catch (SQLException e1) {
					e1.printStackTrace();
				}
				conn = null;
			}
		}
	}

	public void log(Session session, 
    		long start, long end, EIURI instanceIDuri, EIInstance instance) {
		String instanceID = instanceIDuri.toString();
		// Can these be null? Let's just be safe ...
		String instanceClass = "";
		if (instance.getInstanceClass() != null) {
			instanceClass = instance.getInstanceClass().toString();
		}
		String instanceLabel = instance.getInstanceLabel();
		if (instanceLabel == null) {
			instanceLabel = "";
		}
		String instanceType = "";
		if (instance.getInstanceType() != null) {
			instanceType = instance.getInstanceType().toString();
		}
		super.log(new LogInfoInstance(session, start, end, 
				instanceID, instanceClass, instanceLabel, instanceType,
				buildID, ontologyID));
	}
}
