package org.eaglei.search.provider;

import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.model.EIURI;
import org.eaglei.search.provider.SearchRequest;
import org.eaglei.search.provider.SearchResult;
import org.eaglei.search.provider.SearchResultSet;

/**
 * Trivial implementation of SearchResultMerger that uses result rank. Also trims the results sets
 * to the specified range. 
 * NOTE: the rank across result sets is not in most cases comparable.
 * 
 * @author rfrost
 */
public class SearchResultRankMerger implements SearchProvider {

    private static final Log logger = LogFactory.getLog(SearchResultRankMerger.class);
    private static final boolean DEBUG = logger.isDebugEnabled();
    
    /**
     * System property that controls whether the merger trims results to the paginated range.
     */
    public static final String TRIM_TO_RANGE_PROP = "search.provider.merger.trim.to.range";
    
    /**
     * Default value of the trim to range property.
     */
    public static final String DEFAULT_TRIM_TO_RANGE_PROP = "true";
    
    /**
     * Amount that the result rank is decremented for each position in the order.
     */
    public static final float POSITION_DECREMENT = 0.0001f;
    
    private MultiNodeSearchProvider nestedProvider;
    private boolean trimToRange = true;
    
    public SearchResultRankMerger(final MultiNodeSearchProvider nestedProvider) {
        this.nestedProvider = nestedProvider;
        final String trimToRange = System.getProperty(TRIM_TO_RANGE_PROP, DEFAULT_TRIM_TO_RANGE_PROP);  
        setTrimToRange(Boolean.parseBoolean(trimToRange));
    }

    /**
     * Sets the flag that controls whether the returned SearchResultSet is trimmed to the range specified in the SearchRequest
     * or whether all merged results are returned. In both cases, the SearchRequest range is used when calling the nested SearchProvider.
     * @param trimToRange True to trim merged results.
     */
    public void setTrimToRange(final boolean trimToRange) {
       this.trimToRange = trimToRange; 
    }
    
    /**
     * @see #setTrimToRange(boolean)
     */
    public boolean getTrimToRange() {
        return this.trimToRange;
    }
    
    /* (non-Javadoc)
     * @see org.eaglei.search.provider.SearchProvider#init()
     */
    public void init() throws IOException {
        this.nestedProvider.init();
    }

    @Override
    public SearchCounts count(SearchCountRequest request) throws IOException {
        final Collection<SearchCounts> counts = nestedProvider.count(request);
        // create a single SearchCounts that sums up the counts in the collection
        final SearchCounts aggregatedCounts = new SearchCounts(request.getRequest());
        for (SearchCounts count: counts) {
            for (EIURI uri: count.getClassesForCounts()) {
                final int countForClass = count.getClassCount(uri);
                aggregatedCounts.setClassCount(uri, aggregatedCounts.getClassCount(uri) + countForClass);
            }
        }
        return aggregatedCounts;
    }
    
    /* (non-Javadoc)
     * @see org.eaglei.search.provider.SearchProvider#query(org.eaglei.search.request.SearchRequest)
     */
    public SearchResultSet query(final SearchRequest request) throws IOException {
        // always request the result set starting from the beginning with max adjusted by the original
        // start range, we will trim the beginning for pagination in the merge call
        final int startIndex = request.getStartIndex();
        final int maxResults = request.getMaxResults();
        request.setStartIndex(0); 
        request.setMaxResults(startIndex + request.getMaxResults()); 
        
        // call the next provider
        final Collection<SearchResultSet> results = nestedProvider.query(request);
        
        // reset the start index and max results and merge the results into a single SearchResultSet based on rank
        request.setStartIndex(startIndex);
        request.setMaxResults(maxResults);
        return merge(results, request);
    }
    
    private SearchResultSet merge(final Collection<SearchResultSet> results, final SearchRequest request) {
	    SearchResultSet merged = null;
	    if (results.size() == 1) {
			// if there is only one, no need to merge
            if (DEBUG) {	        
                logger.debug("Collection only included one SearchResultSet, returning that");
            }
			merged = results.iterator().next();
		} else if (results.size() == 0) {
			// if empty, create an empty result set
            if (DEBUG) {
		        logger.debug("Collection was empty returning empty SearchResultSet");
		    }
			final SearchResultSet empty = new SearchResultSet(request);
			empty.setTotalCount(0);
			return empty;
		} else {
			// merge, ordered by rank
			merged = new SearchResultSet(request);
			final List<SearchResult> mergedResults = merged.getResults();
			final SortedSet<SearchResult> sortedResults = new TreeSet<SearchResult>();
            //if (DEBUG) {
			//logger.debug("Merging " + results.size() + " SearchResultSets");                
			//}
            int start = 0;
			for (SearchResultSet resultSet: results) {
			    if (DEBUG) {
	                if (resultSet.getResults().size() > 0) {
	                    logger.debug("Merging result set with " + resultSet.getResults().size() + " results and start " + resultSet.getStartIndex());
	                }
			    }
				merged.setTotalCount(merged.getTotalCount() + resultSet.getTotalCount());
				int i = 1;
				for (SearchResult result: resultSet.getResults()) {
                    // TODO we REALLY need global idf
                    float rank = result.getRank();
                    // decrement by position
				    rank -= i++ * POSITION_DECREMENT;
				    // set the new rank
				    result.setRank(rank);
                    //logger.debug("Adding result with rank: " + result.getRank() + ": " + result.getEntity());
				    if (!sortedResults.add(result)) {
				        // decrement the total count
				        merged.setTotalCount(merged.getTotalCount()-1);
				    }
				}
			}
			for (SearchResult result: sortedResults) {
				merged.getResults().add(result);
			}
            //if (DEBUG) {
			//logger.debug("Size of merged result set: " + merged.getResults().size());
            //}
		}
	    if (getTrimToRange()) {
	        return trimToRange(merged, request);
	    }
	    return merged;
	}
	
	// instead of trimming these result sets, want to instead be caching the full set and returning copies
	// of the page ranges
	private SearchResultSet trimToRange(final SearchResultSet results, final SearchRequest request) {
	    if (DEBUG) {
	        logger.debug("Trimming to range " + request.getStartIndex() + " to " + (request.getStartIndex() + 
	                request.getMaxResults()));
	    }
	    final SearchResultSet trimmed = new SearchResultSet(request);
	    trimmed.setStartIndex(request.getStartIndex());
	    trimmed.setTotalCount(results.getTotalCount());
	    final List<SearchResult> list = results.getResults();

	    // filter out any portions of the results before the requested start index
	    int numToRemoveOnFront = request.getStartIndex() - results.getStartIndex();
	    if (numToRemoveOnFront < 0) {
	        numToRemoveOnFront = 0;
	    }
	    for (int i = numToRemoveOnFront; i < (numToRemoveOnFront+ request.getMaxResults()) && i < list.size(); i++) {
	        SearchResult result = list.get(i);
	        trimmed.getResults().add(result);
	    }
	    return trimmed;
	}

    @Override
    public ClassCountResult getResourceCount(SearchRequest request) {
        // TODO Auto-generated method stub
        return null;
    }

	@Override
	public ClassCountResult getProviderTypeCount(SearchRequest request) {
		// TODO Auto-generated method stub
		return null;
	}

}
