package org.eaglei.search.logging;

import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.Date;
import java.util.UUID;

import org.eaglei.search.provider.SearchRequest;
import org.eaglei.security.Session;

public class AsynchronousLoggerCount extends AsynchronousLogger {

	// TODO: these should be enums
	public static final int ROW_UUID_COL 			= 1;
	public static final int SESSIONID_COL 			= 2;
	public static final int USERID_COL 				= 3;
	public static final int USERINSTITUTION_COL 	= 4;
	public static final int DATE_STRING_COL			= 5;
	public static final int DATE_COL 				= 6;
	public static final int START_COL 				= 7;
	public static final int END_COL 				= 8;
	public static final int DURATION_COL 			= 9;
	public static final int REQUEST_COL 			= 10;
	public static final int BUILDID_COL 			= 11;
	public static final int ONTOLOGYID_COL 			= 12;
	
	public AsynchronousLoggerCount(String tname, String buildVersion,
			String ontVersion) {
		super(tname, buildVersion, ontVersion);
	}

	@Override
	protected void createTable() {
		if (tableName == null) {
			logger.error("Table name was null.");
			conn = null;
			return;
		}
		if (tableName.trim().length() == 0) {
			logger.error("Table name was empty.");
			conn = null;
			return;
		}
		
		String createTableCmd = "CREATE TABLE IF NOT EXISTS " + tableName.trim() + " (" + 
			"op_id varchar(128) character set ascii PRIMARY KEY," +  
			"session_id varchar(128) character set ascii NOT NULL," +  
			"user_id varchar(128) character set ascii NOT NULL," + 
			"user_institution varchar(128) character set ascii NOT NULL," + 
			"date_string varchar(128) character set ascii NOT NULL," +
			"date datetime NOT NULL," +
			"start bigint(20) NOT NULL," + 
			"end bigint(20) NOT NULL," + 
			"duration bigint(20) NOT NULL," + 
			"search_request varchar(256) character set ascii NOT NULL," + 
			"build_id varchar(128) character set ascii NOT NULL," + 
			"ontology_id varchar(128) character set ascii NOT NULL" +
			") ENGINE=MyISAM DEFAULT CHARSET=utf8;";
		
		try {
			PreparedStatement createStmt = conn.prepareStatement(createTableCmd);
			createStmt.execute();
		} catch (SQLException e) {
			logger.error("Asynchronous logger failed to create table " + tableName);
			e.printStackTrace();
			try {
				conn.close();
			} catch (SQLException e1) {
				e1.printStackTrace();
			}
			conn = null;
			return;
		}
	}

	@Override
	protected void fillInInsertCmd(LogInfo entryArg) {
		
		if ( !(entryArg instanceof LogInfoCount) ) {
			logger.error("Incorrect LogInfo type");
			return;
		}
		
		LogInfoCount entry = (LogInfoCount) entryArg;
		
		try {
			UUID opid = UUID.randomUUID();
			long end = entry.getEnd();
			long duration = end - entry.getStart();
			
			String date = dateFormat.format(new Date(end));
			//System.out.println("Date:  " + date);
			
			insertCmd.setString(ROW_UUID_COL, opid.toString());
			insertCmd.setString(SESSIONID_COL, entry.getSessionID());
			insertCmd.setString(USERID_COL, entry.getUserID());
			insertCmd.setString(USERINSTITUTION_COL, entry.getUserInstitution());
			insertCmd.setString(DATE_STRING_COL, date);
			insertCmd.setTimestamp( DATE_COL, new Timestamp(end));
			insertCmd.setLong(START_COL, entry.getStart());
			insertCmd.setLong(END_COL, entry.getEnd());
			insertCmd.setLong(DURATION_COL, duration);
			insertCmd.setString(REQUEST_COL, entry.getRequest().toString());
			insertCmd.setString(BUILDID_COL, entry.getBuildID());
			insertCmd.setString(ONTOLOGYID_COL, entry.getOntologyID());
			
			//System.out.println("INSERT COMMAND:  " + insertCmd.toString());
		} catch (SQLException sqle) {
			logger.error("AsynchronousLogger failed to fill in insert command: " + insertCmd.toString());
		}
	}

	@Override
	protected void setInsertCmd() {
		if (conn != null) {
			String INSERT_CMD = "INSERT INTO " + tableName + " ( " + 
				"op_id, session_id, user_id, user_institution, date_string, date, start, end, " + 
				"duration, search_request, build_id, ontology_id ) VALUES ( " +
				"?, " + // op_id
				"?, " + // session_id
				"?, " + // user_id
				"?, " + // user_institution
				"?, " + // date_string
				"?, " + // date
				"?, " + // start
				"?, " + // end
				"?, " + // duration
				"?, " + // search_request
				"?, " + // build_id
				"?"     // ontology_id
				+ " );";
			try {
				insertCmd = conn.prepareStatement(INSERT_CMD);
			} catch (SQLException e) {
				logger.error("Asynchronous Logger failed to create insert command");
				e.printStackTrace();
				try {
					conn.close();
				} catch (SQLException e1) {
					e1.printStackTrace();
				}
				conn = null;
			}
		}
	}
	
	public void log(Session session, 
    		long start, long end, SearchRequest request) {
		super.log(new LogInfoCount(session, start, end, request, buildID, ontologyID));
	}

}
