package org.eaglei.search.provider.lucene;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.store.RAMDirectory;

import org.eaglei.lexical.lucene.PorterStemAnalyzer;
import org.eaglei.model.EIOntModel;
import org.eaglei.search.datagen.DataGenParams;
import org.eaglei.services.InstitutionRegistry;

/**
 * Factory methods for creating LuceneProviders.
 * 
 * @author frost
 */
public final class LuceneSearchProviderFactory {

    private static final Log logger = LogFactory.getLog(LuceneSearchProviderFactory.class);
    private static final boolean DEBUG = logger.isDebugEnabled();

    /**
     * Name of SearchConfiguration property that holds a directory on classpath where provider looks for lucene index
     */    
    public static final String LUCENE_INDEX_DIR_PROP = "org.eaglei.search.provider.lucene.index.dir";

    /**
     * Creates a LuceneProvider for generated data.
     * @param datagenParams Generation params
     * @return The LuceneProvider
     * @throws IOException Thrown if an error is encountered creating the provider.
     */
    public static LuceneSearchProvider createProviderForGeneratedData(final InstitutionRegistry institutionRegistry, final EIOntModel eagleiOntModel, final DataGenParams datagenParams) throws IOException {
        final Analyzer analyzer = getAnalyzer();
        final Directory dir = getDirectory();
        final LuceneGenerator generator = new LuceneGenerator(eagleiOntModel, analyzer, dir);
        if (datagenParams == null) {
            generator.generate(getDefaultDataGenParams(institutionRegistry)); 
        } else {
            generator.generate(datagenParams);
        }
        return new LuceneSearchProvider(eagleiOntModel, dir, analyzer); 
    }
    
    private static DataGenParams getDefaultDataGenParams(InstitutionRegistry institutionRegistry) {
        final DataGenParams params = DataGenParams.createForAllInstitutions(institutionRegistry.getInstitutions());
        params.setCoreFacilityRange(1, 3);
        params.setResearchFacilityRange(0, 0);
        params.setMaterializeHierarchy(true); 
        return params;
    }
    
    /**
     * Retrieves the Lucene Directory to use as a search index.
     * 
     * @return Lucene Directory.
     * 
     * @throws IOException Thrown if an error is encountered.
     */
    public static Directory getDirectory() throws IOException {
        final String indexDir = System.getProperty(LUCENE_INDEX_DIR_PROP);
        
        // if the index directory property is not null, create an FSDirectory for that
        if (indexDir != null) {
            final File dir = new File(indexDir);
            if (DEBUG) {
                logger.debug("Using Lucene index directory: " + dir.getAbsolutePath());
            }
            if (!dir.exists()) {
                dir.mkdirs();
            } else if (!dir.isDirectory()) {
                logger.error("Specified Lucene index directory is not a directory");
                return new RAMDirectory();       
            }
            return FSDirectory.open(dir);
        } 
        if (DEBUG) {
            logger.debug("Using Lucene RAMDirectory");
        }
        return new RAMDirectory();
    }
    
    /**
     * Creates the Lucene Analyzer to use for searching and indexing.
     * @return Analyzer.
     */
    public static Analyzer getAnalyzer() {
        return new PorterStemAnalyzer();
    }
}
