package org.eaglei.search.provider.lucene;

import java.io.IOException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.store.Directory;

import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIURI;
import org.eaglei.model.EIOntConstants;
import org.eaglei.search.provider.SearchResult;
import org.eaglei.search.datagen.AbstractGenerator;
import org.eaglei.search.datagen.DataGenParams;
import org.eaglei.search.events.ChangeEventPayloadItemImpl;

import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.vocabulary.RDFS;

/**
 * Generates test data (as a Lucene index) using the eagle-i ontology.

 * <ul>
 * <li>Generates resources for a random number of labs for each instance of the
 * university class.
 * <li>For each lab, creates a random number of resources for each subclass of
 * the "top-level" class (e.g. instrument, technique, etc.). The class of each
 * of these resources is randomly selected from the hierarchy.
 * <li>A lucene document is created for each lab and resource with fields according to the
 * schema outlined in LuceneIndexer.
 * </ul>
 * @author frost
 */
public class LuceneGenerator extends AbstractGenerator {

    private static final Log logger = LogFactory.getLog(LuceneGenerator.class);

    private LuceneSearchProviderIndexer indexer;

    public LuceneGenerator(final EIOntModel eagleiOntModel, final Analyzer analyzer, final Directory directory) throws IOException {
        super(eagleiOntModel);
        this.indexer = new LuceneSearchProviderIndexer(eagleiOntModel, analyzer, directory, 
        		new LuceneSearchProvider(eagleiOntModel, directory, analyzer));
    }

    @Override
    public void closeStorage() throws IOException {
        this.indexer.commit();
    }
    
    @Override
    public int generateForInstitution(final String uri, final String label,
            final DataGenParams genParams) throws IOException {
        int total = super.generateForInstitution(uri, label, genParams);
        return total;
        /*
        EIEntity institutionEntity = EIEntity.create(EIURI.create(uri), label);
        EIClass typeClass = eagleiOntModel.getClass(EIURI.create(EagleIOntConstants.UNIVERSITY_CLASS_URI));
        EIEntity typeEntity = typeClass.getEntity();
        
        SearchResult result = new SearchResult(institutionEntity, typeEntity, null, institutionEntity);
        result.addDataTypeProperty(EIURI.create(RDFS.label.getURI()), label);
        this.indexer.indexSearchResult(result, genParams.getMaterializeHierarchy());
        
        return total + 1;
        */
    }
    
    @Override
    public int generateLab(final String institutionURI, final String institutionLabel,
            final String labLabel, final String labURI, final String labURL, final String labTypeURI,
            final DataGenParams genParams) throws IOException {
        int total = super.generateLab(institutionURI, institutionLabel, labLabel, labURI, labURL, labTypeURI, genParams);

        EIEntity labEntity = EIEntity.create(EIURI.create(labURI), labLabel);
        EIEntity institutionEntity = EIEntity.create(EIURI.create(institutionURI), institutionLabel);
        EIClass typeClass = eagleiOntModel.getClass(EIURI.create(labTypeURI));
        EIEntity typeEntity = typeClass.getEntity();
        
        SearchResult result = new SearchResult(labEntity, typeEntity, labEntity, institutionEntity);
        result.addDataTypeProperty(EIURI.create(RDFS.label.getURI()), labLabel);
        this.indexer.indexSearchResult(new ChangeEventPayloadItemImpl(result), genParams.getMaterializeHierarchy());

        return total + 1;
    }
    
    @Override
    public void generateResource(final String institutionURI, final String institutionLabel,
            final String resourceLabel, final String resourceURI, final String resourceTypeURI, final String resourceURL,
            final String labName, final String labURI, final DataGenParams genParams)
            throws IOException {

        EIEntity resourceEntity = EIEntity.create(EIURI.create(resourceURI), resourceLabel);
        EIEntity labEntity = EIEntity.create(EIURI.create(labURI), labName);
        EIEntity institutionEntity = EIEntity.create(EIURI.create(institutionURI), institutionLabel);        
        EIClass typeClass = eagleiOntModel.getClass(EIURI.create(resourceTypeURI));
        EIEntity typeEntity = typeClass.getEntity();
        
        SearchResult result = new SearchResult(resourceEntity, typeEntity, labEntity, institutionEntity);
        result.addDataTypeProperty(EIURI.create(RDFS.label.getURI()), resourceLabel);
        this.indexer.indexSearchResult(new ChangeEventPayloadItemImpl(result), genParams.getMaterializeHierarchy());
    }
}
