package org.eaglei.search.provider.lucene;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.SimpleTimeZone;
import java.util.TimeZone;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.apache.commons.httpclient.SimpleHttpConnectionManager;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIOntConstants;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIProperty;
import org.eaglei.model.EIURI;
import org.eaglei.model.jena.JenaEIOntModel;
import org.eaglei.search.provider.PollingDataHarvester;
import org.eaglei.search.provider.SearchCountRequest;
import org.eaglei.search.provider.SearchCounts;
import org.eaglei.search.provider.SearchRequest;
import org.eaglei.search.provider.SearchResult;
import org.eaglei.search.provider.SearchResultSet;
import org.eaglei.services.InstitutionRegistry;
import org.eaglei.services.repository.ProviderUtils;
import org.eaglei.services.repository.RepositoryHttpConfig;

import com.hp.hpl.jena.query.Query;
import com.hp.hpl.jena.query.QueryExecution;
import com.hp.hpl.jena.query.QuerySolution;
import com.hp.hpl.jena.query.ResultSet;
import com.hp.hpl.jena.query.ResultSetFactory;
import com.hp.hpl.jena.query.ResultSetFormatter;
import com.hp.hpl.jena.rdf.model.Literal;
import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.ModelFactory;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.rdf.model.RDFNode;
import com.hp.hpl.jena.rdf.model.Resource;
import com.hp.hpl.jena.rdf.model.Statement;
import com.hp.hpl.jena.sparql.resultset.SPARQLResult;
import com.hp.hpl.jena.sparql.resultset.XMLInput;
import com.hp.hpl.jena.vocabulary.RDF;

/**
 *
 * @author tbashor
 */
public abstract class AbstractStreamHarvester implements PollingDataHarvester {
    
    private class DataHarvestPoller extends Thread {
        
        DataHarvestPoller() {
            super("DataHarvestPoller");
            setPriority(Thread.MIN_PRIORITY);
            setDaemon(true);
        }

        @Override
        public void run() {
            while (true) {
                try {
                    harvest();
                } catch (Throwable t) {
                    logger.error(t);
                }
                
                try {
                    //TODO make configurable
                    Thread.sleep(10000);
                } catch (InterruptedException ie) {
                    // swallow
                }
            }
        }
        
    }

    private static final Log logger = LogFactory.getLog(AbstractStreamHarvester.class);
    private static final boolean DEBUG = logger.isDebugEnabled();
    
    private static final String SUBJECT= "subject";
    private static final String PREDICATE= "predicate";            
    private static final String OBJECT = "object";

    protected static final DateFormat formatWithTZ = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS z");
    private static final DateFormat format = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS");
    private static final SimpleDateFormat preciseHTTPDate =
        new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss.SSS zzz");
    static {
        format.setTimeZone(TimeZone.getTimeZone("GMT"));
        preciseHTTPDate.setCalendar(new GregorianCalendar(new SimpleTimeZone(0, "GMT"), Locale.getDefault()));
    }
    
    protected final EIOntModel eiOntModel;
    protected final EIEntity institution;
    
    private final List<ResourceChangeListener> listeners = new ArrayList<ResourceChangeListener>();
    private final Set<String> labelURIs;
    private final Set<EIURI> resourceProviderProperties;
    
    protected boolean isInitialized;
    private DataHarvestPoller pollingThread;
        
    /**
     * Creates a new RepositoryHarvester.
     */
    protected AbstractStreamHarvester(final EIOntModel eiOntModel, final EIEntity institution) {
        this.eiOntModel = eiOntModel;
        this.institution = institution;
  
        List<Property> preferredLabelProperties = ((JenaEIOntModel) eiOntModel).getPrefLabelProperties();
        labelURIs = new HashSet<String>();
        for (Property prop: preferredLabelProperties) {
            labelURIs.add(prop.getURI());
        }
        
        resourceProviderProperties = eiOntModel.getResourceProviderProperties();
        
        setInitialized(false);
    }
    
    @Override
    public synchronized void startPolling() {
        if (pollingThread != null) {
            return;
        }
        pollingThread = new DataHarvestPoller();
        pollingThread.start();
    }
    
    public boolean isInitialized() {
        return isInitialized;
    }
    
    public void setInitialized(boolean isInitialized) {
        this.isInitialized = isInitialized;
    }
    
    public abstract String getHarvestInfo();
    
    public void addChangeListener(ResourceChangeListener listener) {
    	listeners.add(listener);
    }
    
    protected void notifyChangeStreamStart() {
    	for (ResourceChangeListener listener : listeners) {
    		listener.onChangeStreamStart(this.institution);
    	}
    }

    protected void notifyChangeEvent(ResourceChangeEvent event) {
    	if (event == null) {
    		return;
    	}
    	for (ResourceChangeListener listener : listeners) {
    		listener.onChangeEvent(event);
    	}
    }

    protected void notifyChangeStreamEnd(Date lastModifiedDate) {
    	for (ResourceChangeListener listener : listeners) {
    		listener.onChangeStreamEnd(this.institution, lastModifiedDate);
    	}
    }

    /**
     * Executes the repository /harvest API and notifies listeners of change information.
     */    
    public abstract void harvest() throws IOException;      
    
    
    protected int generateResourceChangeEvents(InputStream is) {
        int count = 0;
        String currentURI = null;
        Model model = null;
        final ResultSet resultSet = ResultSetFactory.fromXML(is);
        for ( ; resultSet.hasNext() ; )
        {
            QuerySolution soln = resultSet.next() ;
            final Resource resource = soln.getResource(SUBJECT);
            String nextURI = resource.getURI();
            if (currentURI == null) {
                // First data read
                currentURI = nextURI;
                model = ModelFactory.createDefaultModel();
            } else if (! currentURI.equals(nextURI)) {
                // New resource data, send data for previous resource
                notifyChangeEvent(createChangeEventFromModel(currentURI, model));
                count++;
                currentURI = nextURI;
                model = ModelFactory.createDefaultModel();
            }
            final Resource property = soln.getResource(PREDICATE);
            final RDFNode value = soln.get(OBJECT);
            final Property predicate = model.createProperty(property.getURI());
            model.add(model.createStatement(resource, predicate, value));
        }
        if (currentURI != null) {
            // All data read
            notifyChangeEvent(createChangeEventFromModel(currentURI, model));
            count++;
        }
        return count;
    }
        
    private ResourceChangeEvent createChangeEventFromModel(final String uri, final Model model) {
        
        final Resource resource = model.getResource(uri);
           
        // create the EIEntity, initially use URI local name for entity name
        final EIEntity resourceEntity = EIEntity.create(EIURI.create(resource.getURI()), resource.getLocalName());

        // gets the first asserted type in the model
        // TODO handle multiple asserted types
        final Statement type = resource.getProperty(RDF.type);
        String typeURI = null;
        if (type == null) {
            Property isDeleted = model.getProperty(EIOntConstants.IS_DELETED);
            if (isDeleted != null) {
                Statement s = resource.getProperty(isDeleted);
                if (s != null && s.getObject().isLiteral()) {
                    if (((Literal) s.getObject()).getBoolean()) {
                        // has the isDeleted property and the value is true
                        return new ResourceChangeEvent(EIOntConstants.IS_DELETED, resourceEntity, null, null, null);
                    }
                }
            }
        } else {
            typeURI = ((Resource)type.getObject()).getURI();
        }
        if (typeURI == null) {
            logger.error("Resource " + resource.getURI() + " is missing type");
            return null;
        }

        // ensure we have the type in the ontology
        final EIURI typeEIURI = EIURI.create(typeURI);
        // does this have a type from the ontology? 
        if (!eiOntModel.isModelClassURI(typeURI)) {
            logger.error("Resource " + resource.getURI() + " has type " + typeURI + " that does not exist in eagle-i ontology");
            return null;
        }

        // Creates an EIEntity from a URI via look-up against the in-memory Jena ontology model.
        final String typeLabel = eiOntModel.getPreferredLabel(typeEIURI);
        final EIEntity typeEntity = EIEntity.create(typeURI, typeLabel);        
            
        // create the change event
        // TOOD confirm that this resource is actually provided by the institution.
        final ResourceChangeEvent result = new ResourceChangeEvent(null, resourceEntity, typeEntity, null, this.institution);
            
        // update the SearchResult with all of the data (add as either data type or object props)
        final List<Statement> statements = model.listStatements(resource, null, (RDFNode) null).toList();
        for (Statement s: statements) {
            final String propertyURI = s.getPredicate().getURI();
            if (labelURIs.contains(propertyURI)) {
                // Resource label
                String label = s.getLiteral().getLexicalForm();
                final EIEntity newEntity = EIEntity.create(result.getEntity().getURI(), label);
                result.setEntity(newEntity);
                continue;
            }
            final EIURI eiURI = EIURI.create(propertyURI);
            final RDFNode object = s.getObject();
            if (object.isLiteral()) {
                // if the property object is a literal, add that value
                result.addDataTypeProperty(eiURI, ((Literal) object).getLexicalForm());
            } else if (object.isResource()) {
                // if the property is a resource, get the URI
                final Resource value = (Resource) object;
                // save the entity as a result property
                EIURI valueURI = EIURI.create(value.getURI());
                result.addObjectProperty(eiURI, valueURI);
                if (resourceProviderProperties.contains(eiURI)) {
                    result.setLab(valueURI);
                }
            }   
        }
        
        return result;
    }

}