package org.eaglei.search.provider.lucene;

import java.io.IOException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.store.Directory;
import org.eaglei.model.EIOntModel;
import org.eaglei.search.provider.ClassCountResult;
import org.eaglei.search.provider.SearchCountRequest;
import org.eaglei.search.provider.SearchCounts;
import org.eaglei.search.provider.SearchRequest;
import org.eaglei.search.provider.SearchResultSet;
import org.eaglei.search.provider.SearchProvider;

/**
 * Implementation of SearchProvider that dynamically builds a Lucene index (according to the schema defined
 * in LuceneSearchIndexSchema) using the results retrieved from another SearchProvider.
 * @author frost
 */
public final class CompositeLuceneProvider implements SearchProvider {
    
    private static final Log logger = LogFactory.getLog(CompositeLuceneProvider.class);
    private static final boolean DEBUG = logger.isDebugEnabled();

    /*
     * Reference to the the in-memory model of the eagle-i ontology 
     */
    private final EIOntModel eagleiOntModel;
    /*
     * SearchProvider that is queried to populated an embedded Lucene index 
     */
    private final SearchProvider nestedProvider;
    /*
     * SearchProvider for the embedded Lucene index
     */
    private LuceneSearchProvider luceneWrapper;
    /*
     * Lucene indexer
     */
    LuceneSearchProviderIndexer indexer;

    /**
     * Creates a new CompositeLuceneProvider that wraps the specified SearchProvider
     * 
     * @param provider Nested SearchProvider that is queried to populated the Lucene index
     */
    public CompositeLuceneProvider(final EIOntModel eagleiOntModel, final SearchProvider provider,
    		LuceneSearchProviderIndexer indexer) {
        this.eagleiOntModel = eagleiOntModel;
        this.nestedProvider = provider;
        this.indexer = indexer;
        
        // initialize the nested SearchProvider
        
        try {
            if (DEBUG) {
                logger.debug("Initializing nested provider for CompositeLuceneProvider");
            }
            this.nestedProvider.init();
            if (DEBUG) {
                logger.debug("Nested provider initialized");
            }
    
            final Directory dir = indexer.getDirectory();
            final Analyzer analyzer = indexer.getAnalyzer();
            
            this.luceneWrapper = new LuceneSearchProvider(eagleiOntModel, dir, analyzer);
        } catch (IOException e) {
            logger.error(e);
        }
    }
    
    /* (non-Javadoc)
     * @see org.eaglei.search.provider.SearchProvider#init()
     */
    public void init() throws IOException {

    }
    
    public void setModelUsageCache(DataSetAnalyzer modelUsageCache) {
        luceneWrapper.setModelUsageCache(modelUsageCache);
    }
    
    /* (non-Javadoc)
     * @see org.eaglei.search.provider.SearchProvider#query(org.eaglei.search.request.SearchRequest)
     */
    public SearchResultSet query(final SearchRequest request) throws IOException {   	
        return this.luceneWrapper.query(request);
    }
    
    /* (non-Javadoc)
     * @see org.eaglei.search.provider.SearchProvider#count(org.eaglei.search.provider.SearchCountRequest)
     */
    public SearchCounts count(SearchCountRequest request) throws IOException {
        return this.luceneWrapper.count(request);
    }

    @Override
    public ClassCountResult getResourceCount(SearchRequest request) {
        return this.luceneWrapper.getResourceCount(request);
    }
}
