package org.eaglei.search.provider.lucene.suggest;

import java.io.IOException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.PhraseQuery;
import org.apache.lucene.search.ScoreDoc;
import org.apache.lucene.search.TopDocs;
import org.apache.lucene.store.Directory;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIOntConstants;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIURI;
import org.eaglei.search.provider.lucene.ResourceChangeEvent;
import org.eaglei.search.provider.lucene.ResourceChangeListener;

public class LuceneAutoSuggestIndexer implements LuceneAutoSuggestIndexSchema,ResourceChangeListener {

    private static final Log logger = LogFactory.getLog(LuceneAutoSuggestIndexer.class);
    private static final boolean DEBUG = logger.isDebugEnabled();
    private static final DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS");

    private int count;
    private EIOntModel eiOntModel;
    private Analyzer analyzer;
    private Directory directory;
	// Flag indicating, don't bother querying the index if it's empty
    private boolean indexEmpty;
    
    private IndexWriter iwriter;
    private HashMap<EIURI, List<Document>> mapURIToDocuments;
    private Set<EIEntity> resourceRoots = new HashSet<EIEntity>();
    
    private static final EIURI RESOURCE_PROVIDER_PROPERTY = EIURI.create(EIOntConstants.PG_RELATED_RESOURCE_PROVIDER);
    
    public LuceneAutoSuggestIndexer(final EIOntModel eiOntModel, final Analyzer analyzer, final Directory directory) {
        this.eiOntModel = eiOntModel;
        this.analyzer = analyzer;
        this.directory = directory;
		for (EIClass c : eiOntModel.getClassesInGroup(EIOntConstants.CG_DATA_MODEL_CREATE)) {
			resourceRoots.add(c.getEntity());
		}
		this.indexEmpty = true;  // assume it is empty on construction (for now)
    }

    @Override
    public void onChangeStreamStart(EIEntity institution) {
    	if (!indexEmpty) {
    		return;
    	}
        logger.debug("onChangeStreamStart");
        count = 0;
        try {
            iwriter = new IndexWriter(directory, analyzer, true, IndexWriter.MaxFieldLength.LIMITED);
            mapURIToDocuments = new HashMap<EIURI, List<Document>>();
        } catch(IOException e) {
            logger.error("Error creating lucene IndexWriter" + e);
        }
    }

    @Override
    public void onChangeEvent(ResourceChangeEvent event) {
    	if (!indexEmpty) {
    		return;
    	}
    	if (event == null) {
    		logger.error("Null change event notification");
    		return;
    	}
    	count++;
    	if (count % 100 == 0) {
    		logger.debug("Received " + count + " change events...");
    	}
    	if (event.getChangeId() != null && event.getChangeId().equals(EIOntConstants.IS_DELETED)) {
    		logger.warn("Unhandled Delete event for " + event.getEntity());
    		return;
    	}
        createInstanceDocuments(event);
    }

    @Override
    public void onChangeStreamEnd(EIEntity institution, Date lastModifiedDate) {
    	if (!indexEmpty) {
    		return;
    	}
        logger.debug("onChangeStreamEnd: num change events " + count + 
        		" last modifed: " + dateFormat.format(lastModifiedDate));
        try {
            for (EIURI uri : mapURIToDocuments.keySet()) {
                List<Document> docs = mapURIToDocuments.get(uri);
                for (Document doc : docs) {
                	iwriter.addDocument(doc);
                }
            }
            if (indexEmpty && mapURIToDocuments.size() > 0) {
            	indexEmpty = false;
            }
            iwriter.optimize();
            iwriter.close();
            logger.debug("wrote " + mapURIToDocuments.size() + " Documents to index.");
        } catch (CorruptIndexException e) {
            logger.error("Error writing change event Documents" + e);
        } catch (IOException e) {
            logger.error("Error writing change event Documents" + e);
        }
        iwriter = null;
        mapURIToDocuments = null;
    }
    
    private List<Document> getDocuments(EIURI uri, boolean createOnMiss) {
    	// First check cache
    	List<Document> docs = mapURIToDocuments.get(uri);
        if (docs != null) {
            return docs;
        }
        // Then the index
        docs = getDocumentsFromIndex(uri);
        if (docs != null) {
            return docs;
        }  
        // Unrecognized uri, create if asked
        if (!createOnMiss) {
        	return null;
        }
     	EIClass typeClass = eiOntModel.getClass(uri);
    	if (typeClass != null) {
            // Note: need to put unpopulated doc in map immediately to short circuit
            // potential infinite recursion caused by calling setTypeField inside
            // createTypeDocuments()
            docs = new ArrayList<Document>();
            mapURIToDocuments.put(uri, docs);
            createTypeDocuments(uri, docs);
    	} else {
    		// create stub instance URI
    		docs = new ArrayList<Document>();
    		createStubInstanceDocument(uri, docs);
            mapURIToDocuments.put(uri, docs);
    	}
        return docs;
    }
    
    private List<Document> getDocumentsFromIndex(final EIURI uri) {
    	if (indexEmpty) {
    		// Don't bother querying the index if it's empty
    		return null;
    	}
    	try {
	        // create a query 
	        final PhraseQuery propQuery = new PhraseQuery();
	        propQuery.add(new Term(URI_FIELD, uri.toString()));
	        
	        final IndexSearcher searcher = new IndexSearcher(directory, true);
	        searcher.setDefaultFieldSortScoring(true, true);
	        
	        final TopDocs docs = searcher.search(propQuery, 1);
	        if (docs.totalHits == 0) {
	            //logger.error("Did not find " + uri + " in search index");
	            return null;
	        }
	        final List<Document> result = new ArrayList<Document>(docs.scoreDocs.length);
	        for (ScoreDoc scoreDoc : docs.scoreDocs) {
	        	result.add(searcher.doc(scoreDoc.doc));
	        }
	        return result;
    	} catch (IOException e) {
    		logger.error(e);
    		return null;
    	}
    }
    
    private void createStubInstanceDocument(EIURI uri, List<Document> docs) {
        final Document doc = new Document();
        // flag that it's a resource
        doc.add(new Field(IS_INSTANCE_FIELD, Boolean.TRUE.toString(), Field.Store.YES, Field.Index.NO));
        // create a non-indexed field for the URI
        doc.add(new Field(URI_FIELD, uri.toString(), Field.Store.YES, Field.Index.NO));
        docs.add(doc);
    }
    
    /**
     *
     */
    private List<Document> createInstanceDocuments(final ResourceChangeEvent event) {

    	List<Document> docs;
    	Field[] isValueOfFields = null;
    	
    	// Check if this uri has already been encountered.
    	// Possibly as a object property value.
    	// Need to save off the referencing property list
    	docs = mapURIToDocuments.get(event.getEntity().getURI());
    	if (docs != null) {
    		// Assume that the IS_VALUE_OF field list is the same for all synonyms
    		isValueOfFields = docs.get(0).getFields(IS_VALUE_OF_FIELD);
    		mapURIToDocuments.remove(event.getEntity().getURI());
    	}

        // TODO support alternate names
    	docs = new ArrayList<Document>(1);
        final Document doc = new Document();

        // flag that it's a resource
        doc.add(new Field(IS_INSTANCE_FIELD, Boolean.TRUE.toString(), Field.Store.YES, Field.Index.NO));
        // create a non-indexed field for the URI
        String uri = event.getEntity().getURI().toString();
        doc.add(new Field(URI_FIELD, uri, Field.Store.YES, Field.Index.NO));
        String label = event.getEntity().getLabel();
        doc.add(new Field(PREF_LABEL_FIELD, label, Field.Store.YES, Field.Index.NO));
        // create an indexed field with position offsets for computation of 
        // highlights
        doc.add(new Field(LABEL_FIELD, label, Field.Store.YES,
                Field.Index.ANALYZED, Field.TermVector.WITH_POSITIONS_OFFSETS));
        
        if (event.getInstitution() != null) {
            // create a non-indexed field for the providing institution
        	String institutionURI = event.getInstitution().getURI().toString();
        	doc.add(new Field(INSTITUTION_FIELD, institutionURI, Field.Store.YES, Field.Index.NO));
        }
        
        // create type field
        setTypeField(true, doc, event.getType().getURI());
        
        if (isValueOfFields != null) {
        	// Add back any prior references
        	for (Field f : isValueOfFields) {
        		doc.add(f);
        	}
        }
        
        // Index the object property references.
        // Will create stub documents for any instance references
        // that we don't have data for yet.
        for (EIURI propURI : event.getObjectProperties()) {
        	for (EIURI valueURI : event.getObjectProperty(propURI)) {
        		indexPropertyReference(valueURI, propURI);
        	}
        }
        
        // generate a special meta property for resource provider
        if (event.getLab() != null) {
        	EIURI resourceProviderURI = event.getLab();
        	indexPropertyReference(resourceProviderURI, RESOURCE_PROVIDER_PROPERTY);
        }
        
        docs.add(doc);
        mapURIToDocuments.put(event.getEntity().getURI(), docs);
        return docs;
    }
    
    private void indexPropertyReference(EIURI valueResourceURI, EIURI propertyURI) {
		// Note that referenced doc may be a type document or instance document.
		// The list of documents are for synonyms
		List<Document> referencedDocs = getDocuments(valueResourceURI, true);
		for (Document referencedDoc : referencedDocs) {
			boolean alreadyIndexed = false;
			for (String existingProp : referencedDoc.getValues(IS_VALUE_OF_FIELD)) {
				// Check if this property reference has already been indexed.
				if (existingProp.equals(propertyURI.toString())) {
					alreadyIndexed = true;
					break;
				}
			}
			if (alreadyIndexed) {
				// Assume that all docs for this valueURI will have the same
				// IS_VALUE_OF list.
				break;
			}
			referencedDoc.add(new Field(IS_VALUE_OF_FIELD, propertyURI.toString(), Field.Store.YES, Field.Index.NOT_ANALYZED)); 
		}
    }

    /*
     * Populate the type field in a Document.  Same for both instances and type documents.
     */
    private void setTypeField(boolean isInstance, Document doc, EIURI typeURI) {
    	EIEntity rootType = null;
        //   the asserted type
        doc.add(new Field(TYPE_FIELD, typeURI.toString(), Field.Store.YES, Field.Index.NOT_ANALYZED));
        getDocuments(typeURI, true);
        if (rootType ==  null && isInstance) {
        	// If an instance, then just use the asserted type
        	EIClass type = eiOntModel.getClass(typeURI);
        	rootType = type.getEntity();
        }
        //   the super types
        List<EIClass> resourceSuperClasses = eiOntModel.getSuperClasses(typeURI);
        for (EIClass superClass : resourceSuperClasses) {
            EIURI superURI = superClass.getEntity().getURI();
            doc.add(new Field(TYPE_FIELD, superURI.toString(), Field.Store.YES, Field.Index.NOT_ANALYZED));
            getDocuments(superURI, true);
            if (rootType == null && resourceRoots.contains(superClass.getEntity())) {
            	rootType = superClass.getEntity();
            }
        }
        if (rootType == null && resourceSuperClasses.size() > 0) {
        	// Must be an entity that is outside the resource heirarchy
        	rootType = resourceSuperClasses.get(resourceSuperClasses.size()-1).getEntity();
        }
        if (rootType != null) {
        	doc.add(new Field(ROOT_TYPE_FIELD, rootType.getLabel(), Field.Store.YES, Field.Index.NOT_ANALYZED));
        }
    }
    
    private void createTypeDocuments(EIURI typeURI, List<Document> docs) {
        EIClass clazz = eiOntModel.getClass(typeURI);
        EIURI uri = clazz.getEntity().getURI();
        for (String label : eiOntModel.getLabels(uri)) {
	        Document doc = new Document();
	
	        // flag that it's not a resource
	        doc.add(new Field(IS_INSTANCE_FIELD, Boolean.FALSE.toString(), Field.Store.YES, Field.Index.NO));
	        // create a non-indexed field for the URI
	        doc.add(new Field(URI_FIELD, uri.toString(), Field.Store.YES, Field.Index.NO));
	        doc.add(new Field(PREF_LABEL_FIELD, clazz.getEntity().getLabel(), Field.Store.YES, Field.Index.NO));
	        // create an indexed field with position offsets for computation of 
	        // highlights
            doc.add(new Field(LABEL_FIELD, label, Field.Store.YES,
                    Field.Index.ANALYZED, Field.TermVector.WITH_POSITIONS_OFFSETS));
            // populate the type field
            setTypeField(false, doc, uri);
            docs.add(doc);
        }
    }
    
}
