package org.eaglei.search.provider.lucene;

import java.io.IOException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.PhraseQuery;
import org.apache.lucene.search.ScoreDoc;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TopDocs;
import org.apache.lucene.store.Directory;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIOntConstants;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIURI;
import org.eaglei.search.harvest.ResourceChangeEvent;
import org.eaglei.search.harvest.ResourceChangeListener;

public abstract class AbstractLuceneIndexer implements ResourceChangeListener, LuceneIndexSchema {

    private static final Log logger = LogFactory.getLog(AbstractLuceneIndexer.class);
    private static final boolean DEBUG = logger.isDebugEnabled();
    private static final DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS");

    private int count;
    protected EIOntModel eiOntModel;
    protected Analyzer analyzer;
    protected Directory directory;
	// Flag indicating, don't bother querying the index if it's empty
    private boolean indexEmpty;
    
    private IndexWriter iwriter;
    // In memory cache of Documents that have been created or changed
    private HashMap<EIURI, List<Document>> mapURIToDocuments;
    private Set<EIEntity> resourceRoots = new HashSet<EIEntity>();
    
    private static final EIURI RESOURCE_PROVIDER_PROPERTY = EIURI.create(EIOntConstants.PG_RELATED_RESOURCE_PROVIDER);
    
    public AbstractLuceneIndexer(final EIOntModel eiOntModel, final Analyzer analyzer, final Directory directory) {
        this.eiOntModel = eiOntModel;
        this.analyzer = analyzer;
        this.directory = directory;
		for (EIClass c : eiOntModel.getClassesInGroup(EIOntConstants.CG_DATA_MODEL_CREATE)) {
			resourceRoots.add(c.getEntity());
		}
		this.indexEmpty = true;  // assume it is empty on construction (for now)
    }

    @Override
    public void onChangeStreamStart(EIEntity institution) {
        logger.debug("onChangeStreamStart: " + institution.getLabel());
        count = 0;
        try {
            iwriter = new IndexWriter(directory, analyzer, IndexWriter.MaxFieldLength.LIMITED);
            mapURIToDocuments = new HashMap<EIURI, List<Document>>();
        } catch(IOException e) {
            logger.error("Error creating lucene IndexWriter" + e);
        }
    }

    @Override
    public void onChangeEvent(ResourceChangeEvent event) {
    	assert (event != null) : "Null change event notification";
    	count++;
    	if (count % 400 == 0) {
    		logger.debug("Received " + count + " change events...");
    	}
    }

    @Override
    public void onChangeStreamEnd(EIEntity institution, Date lastModifiedDate) {
        logger.debug("onChangeStreamEnd: " + institution.getLabel() + "   num change events " + count + 
        		" last modifed: " + dateFormat.format(lastModifiedDate));
        commitDocumentCache();
    }
    
    /**
     * Commits the current in-memory Document cache to the index.
     */
    protected void commitDocumentCache() {
        try {
            for (EIURI uri : mapURIToDocuments.keySet()) {
                List<Document> docs = mapURIToDocuments.get(uri);
            	if (!indexEmpty) {
            		// Delete documents corresponding to this uri
            		// before adding them.
            		deleteDocumentsFromIndex(uri);
            	}
                for (Document doc : docs) {
                	iwriter.addDocument(doc);
                }
            }
            if (indexEmpty && mapURIToDocuments.size() > 0) {
            	indexEmpty = false;
            }
            iwriter.optimize();
            iwriter.close();
            logger.debug("wrote " + mapURIToDocuments.size() + " Documents to index.");
        } catch (CorruptIndexException e) {
            logger.error("Error writing change event Documents" + e);
        } catch (IOException e) {
            logger.error("Error writing change event Documents" + e);
        }
        iwriter = null;
        mapURIToDocuments = null;
    }
    
    /**
     * Gets the Documents associated with this uri.
     * If documents associated with this uri have been previously
     * created, they are guaranteed to be in cache.
     * 
     * @param uri
     * @return
     */
    protected List<Document> getDocuments(EIURI uri) {
    	// First check cache
    	List<Document> docs = mapURIToDocuments.get(uri);
        if (docs != null) {
            return docs;
        }
        // Then the index
    	if (!indexEmpty) {
    		docs = getDocumentsFromIndex(uri);
            if (docs != null) {
            	setDocuments(uri, docs);
                return docs;
            }  
    	}
        return docs;
    }
    
    protected void setDocuments(EIURI uri, List<Document> docs) {
        mapURIToDocuments.put(uri, docs);    	
    }
    
	protected List<Document> getDocumentsFromIndex(EIURI uri) {
		try {
	        final TermQuery uriQuery = new TermQuery(new Term(FIELD_URI, uri.toString()));
	        
	        final IndexSearcher searcher = new IndexSearcher(directory, true);
	        searcher.setDefaultFieldSortScoring(true, true);
	        
	        final TopDocs docs = searcher.search(uriQuery, 1);
	        if (docs.totalHits == 0) {
	            //logger.error("Did not find " + uri + " in search index");
	            return null;
	        }
	        final List<Document> result = new ArrayList<Document>(docs.scoreDocs.length);
	        for (ScoreDoc scoreDoc : docs.scoreDocs) {
	        	result.add(searcher.doc(scoreDoc.doc));
	        }
	        return result;
		} catch (IOException e) {
			logger.error(e);
			return null;
		}
	}
    
    protected List<Document> deleteDocuments(EIURI uri) {
    	List<Document> docs = getDocuments(uri);
    	mapURIToDocuments.remove(uri);
    	if (!indexEmpty) {
    		deleteDocumentsFromIndex(uri);
    	}
    	return docs;
    }
         
    protected void deleteDocumentsFromIndex(final EIURI uri) {
    	try {
	    	assert(iwriter != null);
	        final TermQuery uriQuery = new TermQuery(new Term(FIELD_URI, uri.toString()));
	        iwriter.deleteDocuments(uriQuery);        
            logger.debug("deleted " + uri.toString() + " from index.");
		} catch (IOException e) {
			logger.error(e);
		}
    }
    
}
