package org.eaglei.search.provider.lucene;

import org.eaglei.model.EIOntConstants;

/**
 * Holds information about the schema for the Lucene index used to support search over eagle-i RDF resource data.
 * 
 * This index is populated from the RDF data by creating a Lucene document for each eagle-i "resource" (or subject 
 * RDF resource whose type is a subclass of the "resource" types). Each of these documents contains fields for 
 * the direct resource data type and object properties as well as fields corresponding to indirect relationships: inferred types 
 * and textual descriptions of resource related by object properties. 
 * 
 * The following fixed field names are supported (see Javadocs below for details):
 * <ul>
 * <li>uri
 * <li>resourceFlag
 * <li>related
 * <li>pref_text
 * <li>text
 * <li>inferredType 
 * <li>institution_uri 
 * <li>institution_label 
 * </ul>
 *  
 * The following variable field names are supported:
 * <ul>
 * <li><property_uri>: The text value for each object and datatype property is stored in a multi-valued field using 
 *                     the property URI. Stored, indexed and analyzed. For data type properties, the value is the
 *                     property value. For object properties, the value is the label of the property resource.
 * <li><property_uri>_uri: For object properties, this holds the resource URI in a multi-valued field. 
 *                     Stored and indexed.
 * </ul>
 * @author frost
 */
public class LuceneSearchIndexSchema {
    
    /**
     * The unique URI of the resource. Stored and indexed but not analyzed.
     */
    public final static String URI = "uri";
    /**
     * Field that holds a boolean value indicating whether the resource is an eagle-i resource
     * (i.e. subclass of a top-level resource) or an instance of a non-resource class (e.g. technique). 
     * Stored and indexed but not analyzed.
     */
    public final static String RESOURCE_FLAG = "resourceFlag";
    /**
     * Multi-valued field that holds the URIs of all object properties. 
     * Stored and indexed but not analyzed.
     */
    public final static String RELATED = "related";        
    /**
     * Multi-valued field that holds the lexical form of all preferred datatype properties.
     * A boost is given to this field. Stored, indexed and analyzed. 
     */
    public final static String PREF_TEXT = "pref_text";
    /**
     * Multi-valued field that holds the lexical form of preferred datatype properties, non-preferred datatype 
     * properties and preferred datatype props of resource related by object props. Stored, indexed and analyzed.
     */
    public final static String TEXT = "text";
    /**
     * Multi-valued field that holds the URIs of all inferred classes. 
     * Stored and indexed but not analyzed.
     */
    public final static String INFERRED_TYPE= "inferredType";
    /**
     * Holds the URI of the institution. Stored and indexed but not analyzed.
     */
    public final static String INSTITUTION_URI = "institution_uri";
    /**
     * Holds the name of the institution. Stored, indexed and analyzed.
     */
    public final static String INSTITUTION_LABEL = "institution_label";
    
    /**
     * Suffix for fields that are used to represent object properties.
     */
    public final static String OBJECT_URI_POSTFIX= "_uri";
    
    /**
     * Boost value for fields that should be should be given a lower than normal boost.  
     */
    public final static float LOW_BOOST = 0.5f;
    /**
     * Standard boost
     */
    public final static float STANDARD_BOOST = 1.0f;
    /**
     * Medium boost
     */
    public final static float MEDIUM_BOOST = 4.0f;
    /**
     * High boost
     */    
    public final static float HIGH_BOOST = 7.0f;
    /**
     * Highest boost
     */    
    public final static float HIGHEST_BOOST = 10.0f;
    
    /**
     * Flag that controls whether the preferred labels of object properties are indexed/updated
     */
    public static final boolean INDEX_OBJECT_PROP_LABELS = true;
    
    /**
     * Checks if the Lucene document field represents a data type or object property
     */
    protected static boolean isPropertyField(final String fieldName) {
        if (fieldName.equals(URI)) {
            return false;
        } else if (fieldName.equals(TEXT)) {
            return false;
        } else if (fieldName.equals(PREF_TEXT)) {
            return false;
        } else if (fieldName.equals(INFERRED_TYPE)) {
            return false;            
        } else if (fieldName.equals(INSTITUTION_LABEL)) {
            return false;            
        } else if (fieldName.equals(INSTITUTION_URI)) {
            return false;
        } else if (fieldName.equals(RELATED)) {
            return false;
        } else if (fieldName.equals(RESOURCE_FLAG)) {
            return false;            
        }
        return true;
    }
    
    /**
     * Checks if the specified property represents a property that associates a resource with a lab
     * @param uri URI of property
     * @return True if it binds a resource to a lab.
     */
    protected static boolean isLabProperty(final String uri) {
        if (uri.equals(EIOntConstants.LOCATED_IN_URI)) {
            return true;
        } else if (uri.equals(EIOntConstants.USED_BY_URI)) {
            return true;
        } else if (uri.equals(EIOntConstants.SERVICE_PROVIDED_BY_URI)) {
            return true;            
        }
        return false;
    }
}
