package org.eaglei.search.provider.lucene;

import java.io.File;
import java.io.IOException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.store.RAMDirectory;
import org.eaglei.lexical.lucene.PorterStemAnalyzer;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIOntModel;
import org.eaglei.services.InstitutionRegistry;
import org.eaglei.services.repository.RepositoryHttpConfig;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Lazy;

@Configuration
public class LuceneSearchConfig {

	private static final Log logger = LogFactory.getLog(LuceneSearchConfig.class);
	private static final boolean DEBUG = logger.isDebugEnabled();
	    
    private @Autowired EIOntModel eagleiOntModel;
    private @Autowired InstitutionRegistry institutionRegistry;
    
    @Bean
    public Directory luceneDirectory() {
        try {
            final String indexDir = System.getProperty(LuceneSearchProviderProperties.LUCENE_INDEX_DIR_PROP);
            
            // if the index directory property is not null, create an FSDirectory for that
            if (indexDir != null) {
                final File dir = new File(indexDir);
                if (DEBUG) {
                    logger.debug("Using Lucene index directory: " + dir.getAbsolutePath());
                }
                if (!dir.exists()) {
                    dir.mkdirs();
                } else if (!dir.isDirectory()) {
                    logger.error("Specified Lucene index directory is not a directory");
                    return new RAMDirectory();
                }
                return FSDirectory.open(dir);
            } 
            
            Directory ram = new RAMDirectory();
            
            if (DEBUG) {
                logger.debug("Using Lucene RAMDirectory " + ram.toString());
            }
            return ram;
		} catch (IOException e) {
	        logger.error("Exception getting Lucene directory, using RAMDirectory. ", e);
	        return new RAMDirectory();
		}
    }
    
    @Bean
    public Analyzer luceneAnalyzer() {
    	return new PorterStemAnalyzer();
    }
    
    @Lazy @Bean
    public LuceneSearchProviderIndexer luceneSearchIndexer() throws IOException {
        return new LuceneSearchProviderIndexer(eagleiOntModel, luceneAnalyzer(), luceneDirectory());
    }
    
    @Lazy @Bean
    public RepositoryHarvester harvester() throws IOException {
        EIEntity institution = institutionRegistry.getInstitution();
        RepositoryHttpConfig repoConfig = institutionRegistry.getRepositoryHttpConfig(institution.getURI().toString());
        return new RepositoryHarvester(eagleiOntModel, institution, repoConfig, luceneSearchIndexer());
    }
    
}
