package org.eaglei.search.provider.lucene.search;

import java.io.File;
import java.io.IOException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.store.RAMDirectory;
import org.eaglei.lexical.lucene.PorterStemAnalyzer;
import org.eaglei.model.EIOntModel;
import org.eaglei.search.harvest.PollingDataHarvester;
import org.eaglei.search.provider.lucene.harvest.DataHarvesterConfig;
import org.eaglei.search.provider.lucene.search.LuceneSearchConfig;
import org.eaglei.services.InstitutionRegistry;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Lazy;


@Configuration
@Import(DataHarvesterConfig.class)
public class LuceneSearchConfig {

	private static final Log logger = LogFactory.getLog(LuceneSearchConfig.class);

    private @Autowired EIOntModel eagleiOntModel;
    private @Autowired InstitutionRegistry institutionRegistry;
  
    @Autowired
    private PollingDataHarvester dataHarvester;

    public @Bean Directory searchDirectory() {
        try {
            String indexDir = System.getProperty("org.eaglei.lucene.dir");
            
            // if the index directory property is not null, create an FSDirectory for that
            if (indexDir != null) {
            	indexDir += "/search";
            	indexDir += "/" + institutionRegistry.getLocalNodeId();
                final File dir = new File(indexDir);
                logger.debug("Using Lucene index directory: " + dir.getAbsolutePath());
                if (dir.exists()) {
                	// Delete existing dir
                	// TODO parameterize
                	File[] contents = dir.listFiles();
                	if (contents != null) {
                		for (File f : contents) {
                        	if (!f.delete()) {
        	                    throw new RuntimeException("Error deleting lucene directory: " + f.getAbsolutePath());
                        	}
                		}
                	}
                	if (!dir.delete()) {
	                    throw new RuntimeException("Error deleting lucene directory: " + dir.getAbsolutePath());
                	}
                    logger.debug("Deleted Lucene index directory: " + dir.getAbsolutePath());
                }
                if (!dir.mkdirs()) {
                    throw new RuntimeException("Error creating lucene directory: " + dir.getAbsolutePath());                	
                }
                return FSDirectory.open(dir);
            } else {
	            Directory ram = new RAMDirectory();
	            logger.debug("Using Lucene RAMDirectory " + ram.toString());
	            return ram;
            }
		} catch (IOException e) {
	        logger.error("Exception getting Lucene directory, using RAMDirectory. ", e);
	        return new RAMDirectory();
		}
    }

    public @Bean Analyzer searchIndexAnalyzer() {
    	return new PorterStemAnalyzer();
    }
    
    public @Bean ClassUsageCache classUsageCache() {
        ClassUsageCache classUsageCache = new ClassUsageCache(eagleiOntModel);
        return classUsageCache;
    }
    
    public @Bean LuceneSearchIndexer searchIndexer() {
        LuceneSearchIndexer indexer = new LuceneSearchIndexer(eagleiOntModel, 
        		searchIndexAnalyzer(), searchDirectory(), classUsageCache());
        dataHarvester.addChangeListener(indexer);
        return indexer;
    }
    
    public @Bean LuceneSearchQueryBuilder searchQueryBuilder() {
    	return new LuceneSearchQueryBuilder(eagleiOntModel, searchIndexAnalyzer());
    }
    
    public @Bean LuceneSearchProvider searchProvider() {
    	LuceneSearchProvider provider = new LuceneSearchProvider(eagleiOntModel, institutionRegistry, 
    			searchDirectory(), searchIndexAnalyzer(), searchQueryBuilder(), classUsageCache());
    	return provider;
    }

}
