package org.eaglei.search.provider.ncbi;

import java.io.IOException;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.eaglei.search.provider.ncbi.NCBIEUtils.DocSummary;
import org.eaglei.search.provider.ncbi.NCBIEUtils.ESearchResult;

import org.w3c.dom.Document;

/**
 * Abstract class that provides search support via eUtils for a specific NCBI Entrez database.
 * 
 * Requirement: no dependencies on other eagle-i classes are allowed
 *
 * @author rfrost
 */
public abstract class NCBIDBProvider {

    protected static final Log logger = LogFactory.getLog(NCBIDBProvider.class);
    protected static final boolean DEBUG = logger.isDebugEnabled();

    /**
     * Name of tool for eagle-i that is passed to NCBI eUtils.
     */
    protected static final String EAGLEI_TOOL = "eaglei";
    
    protected final String entrezDB;

    /**
     * Creates a new NCBIDBProvider that executes queries against NCBI.
     */
    public NCBIDBProvider(final String entrezDB) {
        assert entrezDB != null;
        this.entrezDB = entrezDB;
    }
    
    public ESearchResult query(final String query, final int startIndex, final int maxResults) throws IOException {
        final String searchURL = NCBIEUtils.buildSearchURL(query, this.entrezDB, EAGLEI_TOOL, startIndex, maxResults); 
        if (DEBUG) {
            logger.debug("Querying NCBI eSearch service at " + searchURL);
        }
     
        Document doc = NCBIEUtils.executeNCBIRequest(searchURL);
        ESearchResult result = NCBIEUtils.parseESearchResult(doc);
        if (!result.ids.isEmpty()) {
            final String summaryURL = NCBIEUtils.buildSummaryURL(result, this.entrezDB, EAGLEI_TOOL);
            if (DEBUG) {
                logger.debug("Querying NCBI eSummary at " + summaryURL);
            }            
            doc = NCBIEUtils.executeNCBIRequest(summaryURL);
            //if (DEBUG) {
            //logger.debug("Received doc: " + NCBIEUtils.serializeDocument(doc));
            //}            
            final List<DocSummary> docs = NCBIEUtils.parseESummaryResult(doc);
            if (DEBUG) {
                logger.debug("Retrieved " + docs.size() + " summary documents");
            }            
            result.docs.addAll(docs);
        }        
        return result;
    }
}
