package org.eaglei.search.provider.ncbi;

import java.io.IOException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Contains logic for using NCBI ESearch.
 *  
 * @author rfrost
 */
public final class ESearch extends EUtils {

    private static final Log logger = LogFactory.getLog(ESearch.class);
    private static final boolean DEBUG = logger.isDebugEnabled();
    private static final String EUTILS_SEARCH_URL = EUTILS_URL + "esearch.fcgi?";

    /**
     * Executes an NCBI search.
     */
    public static ESearchResult search(final String query, final String db, final String tool, final int startIndex, final int maxResults) throws IOException {
        final String searchURL = ESearch.buildSearchURL(query, db, tool, startIndex, maxResults); 
        if (DEBUG) {
            logger.debug("Querying NCBI eSearch service at " + searchURL);
        }
     
        Document doc = EUtils.executeNCBIRequest(searchURL);
        return new ESearchResult(doc);
    }
    
    /**
     * Builds an ESearchURL
     * @param query The Entrez DB query
     * @param db The target Entrez DB
     * @param tool Source tool
     * @param start Start index
     * @param max Max number of results
     * @return Query URL.
     */
    public static String buildSearchURL(final String query, final String db, final String tool, final int start, final int max) {
        StringBuilder sb = new StringBuilder();
        sb.append(EUTILS_SEARCH_URL);
        sb.append("&" +DB + db);
        sb.append("&" +TOOL + tool);
        sb.append("&" +TERM + URLEncoder.encode(query));
        sb.append("&" +RETMODE + "xml");            
        sb.append("&" +RETSTART + start);                    
        sb.append("&" +RETMAX + max);
        return sb.toString();
    }

    /**
     * Simple representation of an eSearch result and, optionally, the results of a subsequent
     * set of sSummary or eFetch calls.
     */
    public static class ESearchResult {
        /**
         * Total number of results returned
         */
        public int count;
        /**
         * Start of the result set
         */
        public int start;
        /**
         * Max number of results
         */
        public int max;
        /**
         * Translation of the query
         */
        public String queryTranslation;
        /**
         * IDs of result set.
         */
        public List<String> ids = new ArrayList<String>();
        /**
         * XML Document that represent the associated eSummary or eFetch results.
         */
        public Document details;
        
        /**
         * Creates an ESearchResult from an XML Document.
         * @param doc
         * @throws IOException
         */
        public ESearchResult(Document doc) throws IOException {
            final Element eSearchResult = doc.getDocumentElement();
            final NodeList children = eSearchResult.getChildNodes();
            for (int i = 0; i < children.getLength(); i++) {
                final Node node = children.item(i);
                final String nodeName = node.getNodeName();
                if (nodeName.equals(IDLIST)) {
                    final NodeList idNodes = node.getChildNodes();
                    for (int j = 0; j < idNodes.getLength(); j++) {
                        final Node idNode = idNodes.item(j);
                        if (idNode.getNodeName().equals(ID)) {
                            this.ids.add(idNode.getTextContent());
                        }
                    }
                } else if(nodeName.equals("Count")) {
                    this.count = Integer.parseInt(node.getTextContent());
                } else if(nodeName.equals("RetMax")) {            
                    this.max = Integer.parseInt(node.getTextContent());
                } else if(nodeName.equals("RetStart")) {            
                    this.start = Integer.parseInt(node.getTextContent());
                } else if(nodeName.equals("QueryTranslation")) {            
                    this.queryTranslation = node.getTextContent();
                }
            }
        }
    }
}
