package org.eaglei.search.provider.ncbi;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import org.eaglei.search.provider.ncbi.ESearch.ESearchResult;

/**
 * Contains logic for using NCBI EUtils.
 * 
 * @author rfrost
 */
public final class ESummary extends EUtils {

    private static final Log logger = LogFactory.getLog(ESummary.class);
    private static final boolean DEBUG = logger.isDebugEnabled();
    
    public static final String EUTILS_SUMMARY_URL = EUTILS_URL + "esummary.fcgi?";    

    /**
     * Executes an NCBI search and then series of summary calls for the initial page of results.
     */
    public static void getSummaries(final ESearchResult result, final String db, final String tool, final int max) throws IOException {
        if (!result.ids.isEmpty()) {
            final String summaryURL = buildServiceURL(EUTILS_SUMMARY_URL, result.ids, db, max, tool);
            if (DEBUG) {
                logger.debug("Querying NCBI eSummary at " + summaryURL);
            }            
            result.details = EUtils.executeNCBIRequest(summaryURL);
            if (DEBUG) {
                //logger.debug("Received details: " + EUtils.serializeDocument(result.details));
            }            
        }        
    }    

    /**
     * Parses an eSummary XML result and returns the IDs.
     * @param doc XML Document.
     * @return List of IDs
     * @throws IOException
     */
    public static List<ESummaryResult> parseESummaryResult(final Document doc) throws IOException {
    	final List<ESummaryResult> docs = new ArrayList<ESummaryResult>();
    	if (doc != null) {
	    	final Element eSummaryResult = doc.getDocumentElement();
	        final NodeList children = eSummaryResult.getChildNodes();
	        for (int i = 0; i < children.getLength(); i++) {
	            final Node node = children.item(i);
	            final String nodeName = node.getNodeName();
	            if (nodeName.equals("DocSum")) {
	                ESummaryResult docSum = new ESummaryResult(node);
	                docs.add(docSum);
	            } 
	        }
    	}
        return docs;
    }
    
    /**
     * Simple representation of the result (a DocSum) from an eSummary search
     */
    public static class ESummaryResult {
        public String id;
        public Map<String, List<Item>> items = new HashMap<String, List<Item>>();

        public ESummaryResult(Node node) throws IOException {
            final NodeList idNodes = node.getChildNodes();
            for (int j = 0; j < idNodes.getLength(); j++) {
                final Node child = idNodes.item(j);
                if (child.getNodeName().equals("Id")) {
                    this.id = child.getTextContent();
                } else if (child.getNodeName().equals("Item")) {
                    Item item = parseItem(child);
                    if (!this.items.containsKey(item.name)) {
                        this.items.put(item.name, new ArrayList<Item>());
                    }
                    this.items.get(item.name).add(item);
                }
            }
        }
        
        private Item parseItem(Node itemNode) throws IOException {
            String type = itemNode.getAttributes().getNamedItem("Type").getNodeValue();
            Item item = null;
            if (type.equals("List") || type.equals("Structure")) {
                item = new ListItem();
                final NodeList idNodes = itemNode.getChildNodes();
                for (int j = 0; j < idNodes.getLength(); j++) {
                    final Node child = idNodes.item(j);
                    if (child.getNodeName().equals("Item")) {
                        Item childItem = parseItem(child);
                        if (!((ListItem)item).items.containsKey(childItem.name)) {
                            ((ListItem)item).items.put(childItem.name, new ArrayList<Item>());
                        }
                        ((ListItem)item).items.get(childItem.name).add(childItem);                    
                    }
                }
            } else {
                item = new Item();
                item.value = itemNode.getTextContent();
            }
            item.name = itemNode.getAttributes().getNamedItem("Name").getNodeValue(); 
            return item;
        }
    }
    
    public static class Item {
        public String name;
        public String type;
        public String value;
    }
    
    public static class ListItem extends Item {
        public Map<String, List<Item>> items = new HashMap<String, List<Item>>();   
    }
}
