package org.eaglei.search.provider.nif;

import java.io.IOException;
import java.util.Collections;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIURI;
import org.eaglei.search.provider.SearchCountRequest;
import org.eaglei.search.provider.SearchCounts;
import org.eaglei.search.provider.SearchProvider;
import org.eaglei.search.provider.SearchRequest;
import org.eaglei.search.provider.SearchResult;
import org.eaglei.search.provider.SearchResultSet;
import org.eaglei.services.InstitutionRegistry;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Abstract implementation of the eagle-i SearchProvider interface that wraps an NCBIDBProvider.  
 *
 * @author rfrost
 */
public class NIFSearchProvider implements SearchProvider {

    protected static final Log logger = LogFactory.getLog(NIFSearchProvider.class);
    protected static final boolean DEBUG = logger.isDebugEnabled();

    protected final EIOntModel eagleiOntModel;
    protected final InstitutionRegistry institutionRegistry;  
    
    private final EIEntity MOUSE;
    private final EIURI REPOSITORY_URI = EIURI.create("repository");

    /**
     * Creates a new NCBIDBProvider that executes queries against NCBI.
     */
    public NIFSearchProvider(final EIOntModel eagleiOntModel, final InstitutionRegistry registry) {
        assert eagleiOntModel != null;
        assert registry != null;
        this.eagleiOntModel = eagleiOntModel;
        this.institutionRegistry = registry;
        
        MOUSE = eagleiOntModel.getClass(EIURI.create("http://purl.org/obo/owl/NCBITaxon#NCBITaxon_10090")).getEntity();
    }
    
    public void init() throws IOException {
        // no-op
    }
    
    private static final EIURI uri = EIURI.create("foo");
    
    @Override
    public SearchResultSet query(SearchRequest request) throws IOException {
        final SearchResultSet resultSet = new SearchResultSet(request);
        resultSet.setStartIndex(request.getStartIndex());
        if (request.getTerm() == null || request.getTerm().getQuery().length() == 0) {
        	return resultSet;
        }
        final String query = request.getTerm().getQuery(); //getQuery(request);
        Document doc = NIFQueryUtil.search(query, null, null, request.getStartIndex(), request.getMaxResults());
        final Element rootElement = doc.getDocumentElement();
        try {
            int totalCount = Integer.parseInt(rootElement.getAttribute("resultCount"));
            resultSet.setTotalCount(totalCount);
            Node resultsNode = rootElement.getChildNodes().item(0);
            final NodeList resultNodeList = resultsNode.getChildNodes();
            for (int i = 0; i < resultNodeList.getLength(); i++) {
                String repository = null;
                String url = null;
                String description = null;
                final Node resultNode = resultNodeList.item(i);
                final NodeList propertyNodeList = resultNode.getChildNodes();
                for (int j = 0; j < propertyNodeList.getLength(); j++) {
                    final Node propertyNode = propertyNodeList.item(j);
                    String propertyName = propertyNode.getFirstChild().getTextContent();
                    if (propertyName.equals("SITE")) {
                        repository = propertyNode.getLastChild().getTextContent();
                    } else if (propertyName.equals("STRAIN4")) {
                        url = propertyNode.getLastChild().getTextContent();
                    } else if (propertyName.equals("TYPE")) {
                        description = propertyNode.getLastChild().getTextContent();
                    }
                }
                if (repository == null || url == null || description == null) {
                    // throw exception?
                    logger.error(request.toString()+":   NIF mouse result missing expected property. SITE: "+repository+" STRAIN4: "+url+" TYPE: "+description);
                    continue;
                }
                EIEntity entity = EIEntity.create(url, description);
                SearchResult result = new SearchResult(entity, MOUSE, null, null);
                result.addDataTypeProperty(REPOSITORY_URI, repository);
                result.setURL(url);
                if (!"unclassified".equals(description)) {
                    result.setHighlight(description);
                }
                resultSet.getResults().add(result);
            }
        } catch (Exception e) {
            logger.error("Error reading NIF search result.  Request: " + request.toString(), e);
            throw new IOException("Error get search result", e);
        }
        /**/
        /*
        final ESearchResult result = dbProvider.query(query, request.getStartIndex(), request.getMaxResults());
        resultSet.setTotalCount(result.count);
        resultSet.setStartIndex(request.getStartIndex());
        for (ESummaryResult sum: ESummary.parseESummaryResult(result.details)) {
        	final SearchResult searchResult = createSearchResultForDocSummary(sum);
            if (searchResult != null) {
                resultSet.getResults().add(searchResult);
            }
        }
        */
        return resultSet;
    }        

    @Override
    public SearchCounts count(final SearchCountRequest countRequest) throws IOException {
        final SearchRequest request = countRequest.getRequest();
        final SearchCounts counts = new SearchCounts(request);
        /*
        for (EIURI type: countRequest.getCountTypes()) {
            final SearchRequest searchRequest = new SearchRequest(request.toURLParams());
            if (type == null) {
                searchRequest.setBinding(null); // get counts for all types
            } else {
                final SearchRequest.TypeBinding binding = request.getBinding();
                if (binding == null) {
                    searchRequest.setBinding(new SearchRequest.TypeBinding(type));
                } else {
                    searchRequest.getBinding().setType(type);                    
                }
            }
            if (shouldAddResults(searchRequest)) {
                final String query = getQuery(searchRequest);
                final ESearchResult result = dbProvider.query(query, request.getStartIndex(), request.getMaxResults());
                counts.setClassCount(type, result.count);
            }
        }
        */
        return counts;        
    }
    
    /*
    private SearchResult createSearchResultForDocSummary(ESummaryResult sum)
            throws IOException {
        final GeneSummary geneSum = new GeneSummary(sum);
        final SearchResult searchResult = new SearchResult(EIEntity.create(geneSum.url, geneSum.desc.value), entrezGeneEntity, null, locationEntity);
        searchResult.setURL(geneSum.url);
        searchResult.setHighlight("Official Symbol:  "+geneSum.name.value);
        searchResult.addDataTypeProperty(ORGANISM_URI, geneSum.orgname.value);
        searchResult.addDataTypeProperty(AKA_URI, geneSum.otherAliases != null ? geneSum.otherAliases.value : "");
        return searchResult;
    }
    */
    
    private String getQuery(SearchRequest request) throws IOException {
        return getQuery(request, false, Collections.EMPTY_SET, false);
    }    

    /**
     * Gets the type from a SearchRequest.
     * @param request Request
     * @return Type uri
     */
    protected static EIURI getType(SearchRequest request) {
        SearchRequest.TypeBinding binding = request.getBinding();
        if (binding != null) {
            return binding.getType();
        }
        SearchRequest.Term term = request.getTerm();
        if (term != null && term.getURI() != null) {
            return term.getURI();
        }
        return null;
    }
    
    /**
     * Checks, with inference, if a given type is of another type.
     */
    protected boolean isType(EIURI type, EIURI target) {
        if (type == null || target == null) {
            return false;
        }
        if (type.equals(target)) {
            return true;
        }
        final EIClass eiClass = eagleiOntModel.getClass(target);
        if (eiClass != null && eiClass.hasSubClass()) {
            for (EIClass sub: eagleiOntModel.getSubClasses(target)) {
                if (isType(type, sub.getEntity().getURI())) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * Generic query generation logic
     */
    protected String getQuery(final SearchRequest request, final boolean includeType, final Set<EIURI> typesToIgnore, final boolean includeInstitution) {
        StringBuilder sb = new StringBuilder();
        SearchRequest.Term term = request.getTerm();
        if (term != null) {
            if (term.getQuery() != null) {
                sb.append(term.getQuery() + " ");
            }
            if (term.getURI() != null) {
                for (String label: eagleiOntModel.getLabels(term.getURI())) {
                    sb.append(label + " ");
                }
            }
            // TODO add term URI labels
        } 
        SearchRequest.TypeBinding binding = request.getBinding();
        if (binding != null && includeType) {
            EIURI uri = binding.getType();
            if (!typesToIgnore.contains(uri)) {
                for (String label: eagleiOntModel.getLabels(uri)) {
                    sb.append(label + " ");                
                }
            }
        }

        if (request.getInstitution() != null && includeInstitution) {
            EIEntity entity = this.institutionRegistry.getInstitution(request.getInstitution());
            sb.append(entity.getLabel());
        }

        if (DEBUG) {
            logger.debug("Using NIF query: " + sb.toString());
        }
        
        return sb.toString();
    }        
}
