package org.eaglei.network.driver;

import static org.eaglei.search.common.Serializer.SearchCountsSerializer;
import static org.eaglei.search.common.Serializer.SearchResultSetSerializer;
import static org.spin.tools.Util.guardNotNull;

import java.util.Collection;
import java.util.List;

import org.apache.log4j.Logger;
import org.eaglei.search.common.Serializer;
import org.eaglei.search.provider.SearchCounts;
import org.eaglei.search.provider.SearchResultSet;
import org.spin.tools.Util;

/**
 * 
 * @author Clint Gilbert
 * 
 * Jul 7, 2010
 *
 * Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 *
 */
public final class ResultSerializer<T>
{
    private static final Logger log = Logger.getLogger(ResultSerializer.class);
    private static final boolean INFO = log.isInfoEnabled();
    private static final boolean DEBUG = log.isDebugEnabled();
    
    public static final ResultSerializer<SearchResultSet> SearchResultSet = new ResultSerializer<SearchResultSet>(SearchResultSetSerializer);
    
    public static final ResultSerializer<SearchCounts> SearchCounts = new ResultSerializer<SearchCounts>(SearchCountsSerializer);
    
    private final Serializer<T> serializer;
    
    private ResultSerializer(final Serializer<T> serializer)
    {
        super();
        
        guardNotNull(serializer);
        
        this.serializer = serializer;
    }
    
    public Collection<T> deserialize(final Collection<String> results)
    {
        guardNotNull(results);
        
        final List<T> unmarshalled = Util.makeArrayList(results.size());

        for(final String json : results)
        {
            if(json != null)
            {
                // TODO: error handling
                // TODO: is Gson thread-safe?
                try
                {
                    final T resultSet = serializer.deserialize(json);

                    unmarshalled.add(resultSet);
                }
                catch(org.eaglei.search.common.SerializationException e)
                {
                    log.error("Error deserializing result, proceeding with others: ", e);

                    if(DEBUG)
                    {
                        log.debug("Raw results that couldn't be deserialized: '" + json + "'");
                    }
                }
            }
            else
            {
                log.warn("Result is null, skipping deserialization and proceeding with others");

                unmarshalled.add(null);
            }
        }

        if(INFO)
        {
            log.info("Got results from " + unmarshalled.size() + " repositories");
        }

        return unmarshalled;
    }
}
