package org.eaglei.network.driver;

import static org.eaglei.search.common.Serializer.SearchCountRequestSerializer;
import static org.eaglei.search.common.Serializer.SearchRequestSerializer;
import static org.spin.tools.Util.guardNotNull;
import static org.spin.tools.config.EndpointType.SOAP;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;

import org.apache.log4j.Logger;
import org.eaglei.search.common.Serializer;
import org.eaglei.search.provider.MultiNodeSearchProvider;
import org.eaglei.search.provider.SearchCountRequest;
import org.eaglei.search.provider.SearchCounts;
import org.eaglei.search.provider.SearchRequest;
import org.eaglei.search.provider.SearchResultSet;
import org.spin.node.NodeException;
import org.spin.node.acknack.AckNack;
import org.spin.node.connector.NodeConnector;
import org.spin.query.message.agent.AgentException;
import org.spin.query.message.agent.Querier;
import org.spin.query.message.agent.Querier.Credentials;
import org.spin.query.message.agent.TimeoutException;
import org.spin.query.message.identity.IdentityService;
import org.spin.query.message.identity.IdentityServiceException;
import org.spin.query.message.identity.local.LocalAuthEntry;
import org.spin.query.message.identity.local.LocalIdentityService;
import org.spin.tools.Durations;
import org.spin.tools.config.AgentConfig;
import org.spin.tools.config.ConfigException;
import org.spin.tools.config.ConfigTool;
import org.spin.tools.config.EndpointConfig;
import org.spin.tools.crypto.signature.Identity;

/**
 * 
 * @author Clint Gilbert
 * @author Ricardo DeLima
 * 
 *         Jan 27, 2010
 * 
 *         Center for Biomedical Informatics (CBMI)
 * 
 */
public final class SpinMultiNodeSearchProvider implements MultiNodeSearchProvider
{
    private static final Logger log = Logger.getLogger(SpinMultiNodeSearchProvider.class);

    private static final boolean INFO = log.isInfoEnabled();

    private static final boolean DEBUG = log.isDebugEnabled();

    static final Float DefaultPollingFrequency = Float.valueOf(5.0F); // 5Hz
                                                                      // (pretty
                                                                      // fast)

    static final String DefaultPeerGroup = "EAGLE-I-TEST";

    static final Credentials DefaultCredentials = new Credentials("eagle-i.org", "eagle-i-search-app", "eagle-i-search-app");

    static enum Props
    {
        ProviderSpinNetworkUrlPropertyName("org.eaglei.search.provider.spin.network.url"), ProviderSpinQueryTypePropertyName("search.provider.spin.query.type");

        private Props(final String propKey)
        {
            this.propKey = propKey;
        }

        public final String propKey;
    }

    private final Querier querier;

    public SpinMultiNodeSearchProvider() throws IOException
    {
        this(System.getProperty(Props.ProviderSpinNetworkUrlPropertyName.propKey));
    }

    public SpinMultiNodeSearchProvider(final String entryPointNodeURL) throws IOException
    {
        this(makeAgentConfig(guardEntryPointURLIsNotNull(entryPointNodeURL)), null);
    }

    // TODO: Package-protected to allow access to tests; you almost certainly
    // don't want to call this. -Clint
    SpinMultiNodeSearchProvider(final AgentConfig agentConfig, final NodeConnector nodeConnector) throws IOException
    {
        if(DEBUG)
        {
            log.debug("Creating SPIN Network Querier");
        }
        
        guardNotNull(agentConfig);

        // Trust self-signed certificates with mismatched hostnames
        enableSSLWorkarounds();

        try
        {
            querier = makeQuerier(agentConfig, nodeConnector);
        }
        catch(final Exception e)
        {
            throw new IOException("Error creating Querier: ", e);
        }
        
        if(INFO)
        {
            log.info("Created " + SpinMultiNodeSearchProvider.class.getSimpleName() + " for peer group '" + agentConfig.getPeerGroupToQuery() + "', using entry point '" + agentConfig.getNodeConnectorEndpoint() + "'");
        }
    }

    //Default access, for tests
    static Querier makeQuerier(final AgentConfig agentConfig, final NodeConnector nodeConnector) throws ConfigException, NodeException
    {
        guardNotNull(agentConfig);
        
        if(nodeConnector == null)
        {
            guardNotNull(agentConfig.getNodeConnectorEndpoint(), "Can't have null NodeConnector and no Node endpoint URL; entry point node must be described somewhere");
            
            return new Querier(agentConfig, Holder.identityService);
        }
        
        return new Querier(agentConfig, Holder.identityService, nodeConnector);
    }

    private static void enableSSLWorkarounds()
    {
        SSLUtilities.trustAllHttpsCertificates();
        SSLUtilities.trustAllHostnames();
    }

    //Default access, for tests
    static String guardEntryPointURLIsNotNull(final String entryPointNodeURL) throws IOException
    {
        if(entryPointNodeURL == null)
        {
            throw new IOException("SPIN URL property " + Props.ProviderSpinNetworkUrlPropertyName.propKey + " not set");
        }
        
        return entryPointNodeURL;
    }

    //Default access, for tests
    String getEntryPointURL()
    {
        return querier.getAgent().getAgentConfig().getNodeConnectorEndpoint().getAddress();
    }
    
    @Override
    public synchronized void init() throws IOException
    {
        //NOOP
    }
    
    @Override
    public Collection<SearchCounts> count(final SearchCountRequest countRequest) throws IOException
    {
        return sendQuery(Query.Count, countRequest, SearchCountRequestSerializer, ResultSerializer.SearchCounts, QueryWrapUpStrategy.<SearchCounts>passThrough());
    }

    @Override
    public Collection<SearchResultSet> query(final SearchRequest searchRequest) throws IOException
    {
        return sendQuery(Query.RDF, searchRequest, SearchRequestSerializer, ResultSerializer.SearchResultSet, QueryWrapUpStrategy.SearchResultSet);
    }
    
    private <C, R> Collection<R> sendQuery(final Query query, final C searchRequest, final Serializer<C> requestSerializer, final ResultSerializer<R> resultSerializer, final QueryWrapUpStrategy<R> wrapUpStrategy) throws IOException
    {
        guardNotNull(query);
        guardNotNull(searchRequest);
        guardNotNull(requestSerializer);
        guardNotNull(resultSerializer);
        guardNotNull(wrapUpStrategy);

        if(DEBUG)
        {
            log.debug("making query with type: '" + query.queryType + "'");
        }

        final String serializedRequest = serialize(searchRequest, requestSerializer);

        final AckNack ack = submitQuery(query.queryType, serializedRequest);

        // TODO: would be really nice to avoid re-certifying here
        final Identity identity = certifyIdentity();

        final AnnotatedResults results = receiveResults(ack, identity);
        
        if(DEBUG)
        {
            log.debug("Raw, unmarshalled results: " + results.rawResults);
        }

        return wrapUpStrategy.wrapUp(results.timeoutOccurred, unmarshalResults(results.rawResults, resultSerializer));
    }
    
    private static final class AnnotatedResults
    {
        public final Collection<String> rawResults;
        
        public final boolean timeoutOccurred;

        private AnnotatedResults(final Collection<String> rawResults, final boolean timeoutOccurred)
        {
            super();
            
            this.rawResults = rawResults;
            this.timeoutOccurred = timeoutOccurred;
        }
    }
    
    @SuppressWarnings("synthetic-access")
    private AnnotatedResults receiveResults(final AckNack ack, final Identity identity) throws IOException
    {
        try
        {
            //no timeout
            return new AnnotatedResults(querier.receive(ack, identity), false);
        }
        catch(final TimeoutException e)
        {
            //timeout occurred
            return new AnnotatedResults(getAvailableResultsForTimedOutQuery(ack, identity), true);
        }
        catch(final AgentException e)
        {
            throw new IOException("Error receiving the query '" + ack.getQueryID() + "': ", e);
        }
    }
    
    //Default access, for tests
    static <T> String serialize(final T object, final Serializer<T> serializer) throws IOException
    {
        guardNotNull(serializer);
        
        try
        {
            return serializer.serialize(object);
        }
        catch(final org.eaglei.search.common.SerializationException e)
        {
            throw new IOException("Error serializing SearchRequest: ", e);
        }
    }
    
    private Identity certifyIdentity() throws IOException
    {
        try
        {
            return querier.getIdentityService().certify(DefaultCredentials.getDomain(), DefaultCredentials.getUsername(), DefaultCredentials.getPassword());
        }
        catch(final IdentityServiceException e)
        {
            throw new IOException("Error sending query: couldn't certify credentials: " + DefaultCredentials, e);
        }
    }

    private Collection<String> getAvailableResultsForTimedOutQuery(final AckNack ack, final Identity identity) throws IOException
    {
        guardNotNull(ack);
        guardNotNull(identity);

        try
        {
            return querier.decryptResults(querier.getAgent().getResultNoDelete(ack.getQueryID(), identity));
        }
        catch(final Exception e)
        {
            throw new IOException("Error getting results of timed-out query '" + ack.getQueryID() + "'", e);
        }
    }

    private AckNack submitQuery(final String queryType, final String serializedSearchRequest) throws IOException
    {
        try
        {
            return querier.queryAsync(queryType, DefaultCredentials, serializedSearchRequest);
        }
        catch(final AgentException e)
        {
            throw new IOException("Error submitting the Query: (queryType: '" + queryType + "')", e);
        }
    }

    private static <T> Collection<T> unmarshalResults(final Collection<String> results, final ResultSerializer<T> serializer)
    {
        guardNotNull(results);
        guardNotNull(serializer);

        return serializer.deserialize(results);
    }

    private static final class Holder
    {
        private Holder()
        {
            super();
        }
        
        @SuppressWarnings("synthetic-access")
        static final IdentityService identityService = makeIdentityService();
    }

    private static IdentityService makeIdentityService()
    {
        final LocalAuthEntry defaultAuthEntry = new LocalAuthEntry(DefaultCredentials.getDomain(), DefaultCredentials.getUsername(), DefaultCredentials.getPassword(), "user");
        
        return new LocalIdentityService(Arrays.asList(defaultAuthEntry));
    }

    private static AgentConfig makeAgentConfig(final String entryPointNodeURL)
    {
        guardNotNull(entryPointNodeURL);

        try
        {
            return ConfigTool.loadAgentConfig();
        }
        catch(final ConfigException e)
        {
            if(INFO)
            {
                log.info("Couldn't load agent.xml from the filesystem or classpath, using defaults.");
            }

            return makeDefaultAgentConfig(entryPointNodeURL);
        }
    }

    public static AgentConfig makeDefaultAgentConfig(final String entryPointNodeURL)
    {
        final AgentConfig config = new AgentConfig();

        config.setMaxWaitTime(Durations.InMilliseconds.oneMinute);
        config.setPollingFrequency(DefaultPollingFrequency);
        config.setPeerGroupToQuery(DefaultPeerGroup);
        config.setNodeConnectorEndpoint(new EndpointConfig(SOAP, entryPointNodeURL));

        return config;
    }
}