package org.eaglei.services.repository;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.apache.commons.httpclient.UsernamePasswordCredentials;
import org.apache.commons.httpclient.auth.AuthScope;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.security.SecurityProvider;
import org.eaglei.security.Session;
import org.eaglei.services.InstitutionRegistry;
import org.eaglei.services.NodeConfig;

import com.hp.hpl.jena.query.QuerySolution;
import com.hp.hpl.jena.query.ResultSet;
import com.hp.hpl.jena.query.ResultSetFactory;
import com.hp.hpl.jena.rdf.model.Literal;
import com.hp.hpl.jena.rdf.model.Resource;

/**
 * 
 * @author Ricardo De Lima
 * @author Lucy Hadden
 * @author Daniela Bourges
 * 
 *         April 11, 2010
 * 
 *         Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 * 
 * 
 * 
 */
public final class RepositorySecurityProvider implements SecurityProvider {

    private static final Log log = LogFactory.getLog(RepositorySecurityProvider.class);
    private static final boolean isDebugEnabled = log.isDebugEnabled();

    private final InstitutionRegistry institutionRegistry;
    private Map<String, Session> mapSessionIdToSession = new HashMap<String, Session>();
    private Map<String, HttpClient> mapSessionIdToHttpClient = new HashMap<String, HttpClient>();

    /**
     * Intitialize from InstitutionRegistry configuration info.
     */
    public RepositorySecurityProvider(InstitutionRegistry institutionRegistry) {
        this.institutionRegistry = institutionRegistry;
    }

    @Override
    public Session logIn(final String institutionId, final String user, final String password) {
        HttpClient userClient = RepositoryHttpConfig.createHttpClient(user, password);
        // null institution id param is OK, on an institutional node
        RepositoryHttpConfig repoConfig = institutionRegistry.getRepositoryHttpConfig(institutionId);
        if (repoConfig == null) {
            log.error("Unrecognized login institution id: "+institutionId);
            return null;
        }

        int status = 0;
        ResultSet results = null;
        String responseBody = null;
        final GetMethod method = new GetMethod(repoConfig.getWhoamiUrl());
        try {
            if (isDebugEnabled) {
                log.debug("Authenticating user " + user + " at " + repoConfig.getWhoamiUrl());
            }

            status = userClient.executeMethod(method);
            responseBody = getStringFromInputStream(method.getResponseBodyAsStream());
            if (status != HttpStatus.SC_OK) {
                if (status == HttpStatus.SC_NOT_FOUND) {
                    log.error("Repo unavailable");
                } else if (status == HttpStatus.SC_UNAUTHORIZED) {
                    // This is the only expected "failure".  Should it be handled differently?
                    log.info("not authorized to get user information (login/whoami) : " + user);
                } else {
                    log.error(responseBody);
                }
                return null;
            }
            
            results = ResultSetFactory.fromXML(responseBody);
            if (results == null) {
                log.error("whoami response body is null");
                return null;
            }
            
            // we are authorized so let's return the user name
            final QuerySolution soln = results.nextSolution();
            final Literal username = soln.getLiteral("username");
            if (username == null) {
                log.error("Could not authenticate, null username in response body: " + user);
                return null;
            }
            final Resource userURI = soln.getResource("uri");
            if (userURI == null) {
                log.error("Could not authenticate, null uri in response body: " + user);
                return null;
            }

            // Repo right now doesn't contain First/Last names so let's just
            // return the username and URI

            if (isDebugEnabled) {
                log.debug("Authenticated user: " + username.getString() + " " + userURI.getURI());
            }

            // I read somewhere that this was a good way to generate session ids.
            //SecureRandom random = new SecureRandom();
            //byte bytes[] = new byte[20];
            //random.nextBytes(bytes);
            //String sessionId = String.valueOf(bytes);
            String sessionId = UUID.randomUUID().toString();
            //List<String> states = getWFStatesList(sessionId, userURI.getURI());
            Session session = new Session(sessionId, institutionId, username.getString(), userURI.getURI());
            mapSessionIdToSession.put(sessionId, session);
            mapSessionIdToHttpClient.put(sessionId, userClient);
            return session;           
        } catch (final Exception e) {
            log.error("problem getting user info " + repoConfig.getWhoamiUrl() + " Message from repo: "
                    + responseBody + "; Exception "
                    + e);
            return null;
        } finally {
            method.releaseConnection();
        }
    }

    @Override
    public void logOut(String sessionId) {
        PostMethod method = null;
        try {
            HttpClient client = mapSessionIdToHttpClient.get(sessionId);
            Session session = mapSessionIdToSession.get(sessionId);
            if (client == null) {
                return;
            }
            if (session == null) {
                return;
            }
            RepositoryHttpConfig repoConfig = institutionRegistry.getRepositoryHttpConfig(session.getInstitutionId());
            method = new PostMethod(repoConfig.getLogoutUrl());
            int status = 0;
            if (isDebugEnabled) {
                log.debug("Trying to logout at " + repoConfig.getLogoutUrl());
            }
            status = client.executeMethod(method);
            // check to see if we succeeded
            if (status == HttpStatus.SC_OK) {
                if (isDebugEnabled) {
                    log.debug("logout succeded");
                }
            } else {
                log.info("Could not logout user: HTTP Status: " + status);
            }
        } catch (final Exception e) {
            log.error("Unexpected error during logout  " + e);
        } finally {
            if (method != null) {
                method.releaseConnection();
            }
            mapSessionIdToHttpClient.remove(sessionId);
            mapSessionIdToSession.remove(sessionId);
        }
    }
    
    //public RepositoryHttpConfig getUserRepositoryHttpClient(String sessionId) {
    //	return mapSessionIdToRepositoryHttpClient.get(sessionId);
    //}

    public HttpClient getUserHttpClient(String sessionId) {
    	return mapSessionIdToHttpClient.get(sessionId);
    }

    private String getStringFromInputStream(InputStream in) throws IOException {
        StringWriter writer = new StringWriter();
        //encoding needs to be explicitly set
        IOUtils.copy(in, writer, "UTF-8");
        return writer.toString();
    }

    @Override
    public Session getSession(String sessionId) {
        return mapSessionIdToSession.get(sessionId);
    }

    @Override
    public boolean isValid(String sessionId) {
        return getSession(sessionId) != null;
    }
}
