/**
 * The eagle-i consortium
 * Harvard University
 * Nov 4, 2010
 */
package org.eaglei.services.repository;

import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIInstanceFactory;
import org.eaglei.model.EIInstanceMinimal;
import org.eaglei.model.EIInstance;
import org.eaglei.model.EIInstanceX;
import org.eaglei.model.EIObjectProperty;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIProperty;
import org.eaglei.model.EIURI;
import org.eaglei.services.InstitutionRegistry;
import org.eaglei.services.NodeConfig;

import com.hp.hpl.jena.query.ResultSet;
import com.hp.hpl.jena.query.ResultSetFactory;

public class RepositoryInstanceProvider {
	private static final Log log = LogFactory.getLog( RepositoryInstanceProvider.class );
	private static final boolean isDebugEnabled = log.isDebugEnabled();
	//TODO these used to be in ProviderMessages - do we pull that out too?
	private static final String NO_SESSION_MESSAGE = "No session information was found.";
	private static final String FAILURE_MESSAGE = "Could not complete operation; repository message is: ";
	
    protected static final String READ_VIEW = "user";
	// string sent as part of http requests
	private static final String FORMAT_VALUE = "application/xml";
	// format string used for received RDF
	private static final String RDF_FORMAT = "RDF/XML";
	
	private EIOntModel eiOntModel;
	private RepositorySecurityProvider securityProvider;
	private EIInstanceFactory instanceFactory;
    // Use connection pooling, will try to keep connections open between uses.
    private MultiThreadedHttpConnectionManager connectionManager = new MultiThreadedHttpConnectionManager();
    private Map<String, RepositoryHttpConfig> mapHostToRepoConfig = new HashMap<String, RepositoryHttpConfig>();
	private Map<String, HttpClient> mapHostToHttpClient = new HashMap<String, HttpClient>();
    /*
     * HttpClient that is reused. It will reuse the underlying connection or recreate if it gets dropped.
     */
    private HttpClient httpclient;
	
	public RepositoryInstanceProvider(EIOntModel eiOntModel, RepositorySecurityProvider provider, InstitutionRegistry registry, EIInstanceFactory instanceFactory) {
	    this.eiOntModel = eiOntModel;
		this.securityProvider = provider;
		this.instanceFactory = instanceFactory;
		
		// Build a lookup table of instance id prefix to search user HttpClient
		for (NodeConfig nodeConfig : registry.getNodeConfigs()) {
		    RepositoryHttpConfig repoConfig = registry.getRepositoryHttpConfig(nodeConfig.getUri());
		    String hostURL = repoConfig.getHostURL();
		    hostURL = hostURL.replace("https", "http");	        
	        httpclient = RepositoryHttpConfig.createHttpClient(repoConfig.getSearchUsername(), repoConfig.getSearchPassword());
	        httpclient.setHttpConnectionManager(connectionManager);
	        
            mapHostToRepoConfig.put(hostURL, repoConfig);
	        mapHostToHttpClient.put(hostURL, httpclient);
		}
	}
	
	public EIInstanceX getInstance(final String sessionID, final EIURI instanceID) throws Exception {
        if ( !securityProvider.isValid( sessionID ) ) {
            log.error( "Invalid Session - request cannot be completed" );
            throw new RepositoryProviderException( NO_SESSION_MESSAGE );
        }

		if ( instanceID == null || instanceID.toString() == null ) {
			log.warn( "Trying to get instance for null EIURI" );
			return null;
		}
		
		String instanceIDStr = instanceID.toString();
		String host = instanceIDStr.substring(0, instanceIDStr.indexOf("/i/") + 1);
        RepositoryHttpConfig repoConfig = mapHostToRepoConfig.get(host);
        if (repoConfig == null) {
		    log.error("Unrecognized resource instance URL:  " + instanceIDStr);
		    throw new Exception("Unrecognized resource instance URL:  " + instanceIDStr);
		}
        HttpClient searchHttpClient = mapHostToHttpClient.get(host);
		
		int status = 0;
		final PostMethod method = new PostMethod( repoConfig.getInstanceUrl() );
		setReadParameters( method );
		method.setParameter( "uri", instanceIDStr );
		if ( isDebugEnabled ) {
			log.debug( "Trying to get instance " + instanceID.toString() );
		}
		try {
			status = searchHttpClient.executeMethod( method );
			if ( status == HttpStatus.SC_OK ) {
				final String response = ProviderUtils.getStringFromInputStream( method.getResponseBodyAsStream() );
				final EIInstance instance = instanceFactory.create( instanceID, response, RDF_FORMAT );
				if ( isDebugEnabled ) {
					log.debug( "got instance contents: " + instance );
				}
				// Get references and add inferred inverses
				List<EIInstanceMinimal> references = getReferencedBy(searchHttpClient, repoConfig, instanceID, instance);
				// Identify term values
				Map<EIURI, String> mapTermToDefinition = getTerms(instance);
				return new EIInstanceX(instance, references, mapTermToDefinition);
			} else {
				log.error( "get instance " + instanceID + " failed with status: " + status );
				throw new RepositoryProviderException(FAILURE_MESSAGE + "get instance, " +status );
			}
		} finally {
			method.releaseConnection();
			// Take the opportunity to close any connections that haven't been used in half an hour.
			// TODO should have reaper thread that does this.
			connectionManager.closeIdleConnections(1800000);
		}
	}

    private List<EIInstanceMinimal> getReferencedBy(HttpClient searchHttpClient, RepositoryHttpConfig repoConfig, final EIURI instanceID, final EIInstance instance) throws Exception {
        int status = 0;
        String q = SPARQLQueryUtil.getReferencedByQuery( instanceID );
        if ( isDebugEnabled ) {
            log.debug( q );
        }
        final PostMethod method = new PostMethod( repoConfig.getSparqlUrl() );
        setReadParameters(method);
        method.setParameter("query", q);
        try {
            status = searchHttpClient.executeMethod( method );
            if ( status == HttpStatus.SC_OK ) {
                final String response = ProviderUtils.getStringFromInputStream( method.getResponseBodyAsStream() );
                final ResultSet resultSet = ResultSetFactory.fromXML(response);
                return SPARQLQueryUtil.createReferencedByInstances(eiOntModel, instance, resultSet);
            } else {
                log.error( "get instance " + instanceID + " failed with status: " + status );
                throw new RepositoryProviderException(FAILURE_MESSAGE + "get instance, " +status );
            }
        } finally {
            method.releaseConnection();
        }

    }
    
    private Map<EIURI, String> getTerms(final EIInstance instance) {
    	Map<EIURI, String> mapTermToDefinition = null;
    	EIClass eiClass = instance.getInstanceClass();
    	List<EIProperty> properties = eiOntModel.getProperties(eiClass.getEntity().getURI());
    	for (Map.Entry<EIEntity, Set<EIEntity>> entry : instance.getObjectProperties().entrySet()) {
    		List<EIEntity> listTermRanges = getTermRanges(entry.getKey(), properties);
    		if (listTermRanges != null) {
    			// At least one range of this property is a term.
    			// Find  out if any of the values are subtypes.
    			for (EIEntity value : entry.getValue()) {
	    			if (isTermType(value, listTermRanges)) {
	    				if (mapTermToDefinition == null) {
	    					mapTermToDefinition = new HashMap<EIURI, String>();
	    				}
	    				mapTermToDefinition.put(value.getURI(), eiOntModel.getClassDefinition(value.getURI()));
	    			}
    			}
    		}
    	}
    	return mapTermToDefinition;
    }
    
    private List<EIEntity> getTermRanges(EIEntity propEntity, List<EIProperty> properties) {
    	for (EIProperty p : properties) {
    		if (p.getEntity().equals(propEntity)) {
    			EIObjectProperty objProperty = (EIObjectProperty) p;
    			List<EIEntity> termRanges = new ArrayList<EIEntity>(objProperty.getRangeList().size());
    			for (EIClass rangeClass : objProperty.getRangeList()) {
    				if (!rangeClass.isEagleIResource()) {
    					termRanges.add(rangeClass.getEntity());
    				}
    			}
    			return termRanges;
    		}
    	}
    	return null;
    }
    
    private boolean isTermType(EIEntity value, List<EIEntity> listTermRanges) {
    	for (EIEntity range : listTermRanges) {
    		if (range.equals(value)) {
    			return true;
    		} else if (eiOntModel.isSubClass(range.getURI(), value.getURI())) {
    			return true;
    		}
    	}
    	return false;
    }
	
	private void setReadParameters(final PostMethod method) {
		method.setParameter( "format", FORMAT_VALUE );
		method.setParameter( "view", READ_VIEW );
		method.setParameter( "noinferred", "true" );
		method.setRequestHeader( "charset", "UTF-8" );
	}
	
    public boolean contactMessage(final String sessionID, final String client_ip, final EIURI instanceID, String label, boolean test_mode, 
            String from_name, String from_email, String subject, String message) throws Exception {
        if ( !securityProvider.isValid( sessionID ) ) {
            log.error( "Invalid Session - request cannot be completed" );
            throw new RepositoryProviderException( NO_SESSION_MESSAGE );
        }

        if ( instanceID == null || instanceID.toString() == null ) {
            log.warn( "Trying to get instance for null EIURI" );
            return false;
        }
        
        String instanceIDStr = instanceID.toString();
        String host = instanceIDStr.substring(0, instanceIDStr.indexOf("/i/") + 1);
        RepositoryHttpConfig repoConfig = mapHostToRepoConfig.get(host);
        if (repoConfig == null) {
            log.error("Unrecognized resource instance URL:  " + instanceIDStr);
            throw new Exception("Unrecognized resource instance URL:  " + instanceIDStr);
        }
        HttpClient searchHttpClient = mapHostToHttpClient.get(host);
        
        int status = 0;
        final PostMethod method = new PostMethod( repoConfig.getContactUrl() );
        method.setRequestHeader( "charset", "UTF-8" );
        method.setRequestHeader("format", "application/x-www-form-urlencoded" );
        method.setRequestHeader("Referer", repoConfig.getContactUrl().replaceFirst("emailContact", "contact") +
                "?uri="+URLEncoder.encode(instanceIDStr) + 
                "&safe=" + Boolean.toString(test_mode) + 
                "&label=" + URLEncoder.encode(label));
        method.setParameter( "uri", instanceIDStr );
        method.setParameter( "label", label );
        method.setParameter( "test_mode", Boolean.toString(test_mode) );
        method.setParameter( "client_ip", client_ip );
        method.setParameter( "from_name", from_name );
        method.setParameter( "from_email", from_email );
        method.setParameter( "subject", subject );
        method.setParameter( "message", message );
        if ( isDebugEnabled ) {
            log.debug( "Sending contact message " + instanceID.toString() );
        }
        try {
            status = searchHttpClient.executeMethod( method );
            if ( status == HttpStatus.SC_OK ||  status == HttpStatus.SC_MOVED_TEMPORARILY ) {
                return true;
            } else {
                final String response = ProviderUtils.getStringFromInputStream( method.getResponseBodyAsStream() );
                log.error( "send contact " + instanceID + " failed with status: " + status + "\r\n" + response);
                return false;
            }
        } finally {
            method.releaseConnection();
            // Take the opportunity to close any connections that haven't been used in half an hour.
            // TODO should have reaper thread that does this.
            connectionManager.closeIdleConnections(1800000);
        }
        
    }
}
