package org.eaglei.services.repository;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.SimpleHttpConnectionManager;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.security.SecurityProvider;
import org.eaglei.security.Session;
import org.eaglei.services.InstitutionRegistry;

import com.hp.hpl.jena.query.QuerySolution;
import com.hp.hpl.jena.query.ResultSet;
import com.hp.hpl.jena.query.ResultSetFactory;
import com.hp.hpl.jena.rdf.model.Literal;
import com.hp.hpl.jena.rdf.model.Resource;

/**
 * 
 * @author Ricardo De Lima
 * @author Lucy Hadden
 * @author Daniela Bourges
 * @author Ted Bashor
 * 
 *         April 11, 2010
 * 
 *         Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 * 
 * 
 * 
 */
public final class RepositorySecurityProvider implements SecurityProvider {
    
    private static class SessionInternal {
        Session session;
        Long lastAccess;
        HttpClient httpClient;
    }
    
    private class SessionReaper extends Thread {
        
        SessionReaper() {
            setDaemon(true);
        }
        
        public void run() {
            while (true) {
                try {
                    Thread.sleep(SESSION_TIMEOUT);
                } catch (InterruptedException e) {
                    return;
                }
                synchronized(mapSessionIdToSession) {
                    ArrayList<String> sessionList = new ArrayList<String>(mapSessionIdToSession.keySet());
                    for (String sessionId : sessionList) {
                        getSession(sessionId);
                    }
                }
            }
        }
        
    }
    
    // Automatic logout will occur sometime after this amount of idleness.
    // in msec, 30 min default
    private static final Long SESSION_TIMEOUT = Long.getLong("org.eaglei.session.timeout", 1800000);

    private static final Log log = LogFactory.getLog(RepositorySecurityProvider.class);
    private static final boolean isDebugEnabled = log.isDebugEnabled();

    private final InstitutionRegistry institutionRegistry;
    private Map<String, SessionInternal> mapSessionIdToSession = new HashMap<String, SessionInternal>();

    /**
     * Intitialize from InstitutionRegistry configuration info.
     */
    public RepositorySecurityProvider(InstitutionRegistry institutionRegistry) {
        this.institutionRegistry = institutionRegistry;
        SessionReaper reaper = new SessionReaper();
        reaper.start();
        log.info("RepositorySecurityProvider initialized.  Session timeout:  " + SESSION_TIMEOUT/60000 + " min");
    }

    @Override
    public Session logIn(final String institutionId, final String user, final String password) {
        HttpClient userClient = RepositoryHttpConfig.createHttpClient(user, password);
        // For now, this HttpClient is only being used for login/logout
        // so configure to always close connections on releaseConnection
        userClient.setHttpConnectionManager(new SimpleHttpConnectionManager(true));
        // null institution id param is OK, on an institutional node
        RepositoryHttpConfig repoConfig = institutionRegistry.getRepositoryHttpConfig(institutionId);
        if (repoConfig == null) {
            log.error("Unrecognized login institution id: "+institutionId);
            return null;
        }

        int status = 0;
        ResultSet results = null;
        String responseBody = null;
        final GetMethod method = new GetMethod(repoConfig.getWhoamiUrl());
        try {
            if (isDebugEnabled) {
                log.debug("Authenticating user " + user + " at " + repoConfig.getWhoamiUrl());
            }

            status = userClient.executeMethod(method);
            responseBody = getStringFromInputStream(method.getResponseBodyAsStream());
            if (status != HttpStatus.SC_OK) {
                if (status == HttpStatus.SC_NOT_FOUND) {
                    log.error("Repo unavailable");
                } else if (status == HttpStatus.SC_UNAUTHORIZED) {
                    // This is the only expected "failure".  Should it be handled differently?
                    log.info("not authorized to get user information (login/whoami) : " + user);
                } else {
                    log.error(responseBody);
                }
                return null;
            }
            
            results = ResultSetFactory.fromXML(responseBody);
            if (results == null) {
                log.error("whoami response body is null");
                return null;
            }
            
            // we are authorized so let's return the user name
            final QuerySolution soln = results.nextSolution();
            final Literal username = soln.getLiteral("username");
            if (username == null) {
                log.error("Could not authenticate, null username in response body: " + user);
                return null;
            }
            final Resource userURI = soln.getResource("uri");
            if (userURI == null) {
                log.error("Could not authenticate, null uri in response body: " + user);
                return null;
            }

            // Repo right now doesn't contain First/Last names so let's just
            // return the username and URI

            // I read somewhere that this was a good way to generate session ids.
            //SecureRandom random = new SecureRandom();
            //byte bytes[] = new byte[20];
            //random.nextBytes(bytes);
            //String sessionId = String.valueOf(bytes);
            String sessionId = UUID.randomUUID().toString();
            //List<String> states = getWFStatesList(sessionId, userURI.getURI());
            Session session = new Session(sessionId, institutionId, username.getString(), userURI.getURI());
            SessionInternal sessionInternal = new SessionInternal();
            sessionInternal.session = session;
            sessionInternal.lastAccess = new Long(System.currentTimeMillis());
            sessionInternal.httpClient = userClient;
            mapSessionIdToSession.put(sessionId, sessionInternal);
            if (isDebugEnabled) {
                log.debug("Authenticated user: " + sessionInternal.session.getUserName() + " session id: " + sessionId);
            }
            return session;           
        } catch (final Exception e) {
            log.error("problem getting user info " + repoConfig.getWhoamiUrl() + " Message from repo: "
                    + responseBody + "; Exception "
                    + e);
            return null;
        } finally {
            method.releaseConnection();
        }
    }

    @Override
    public void logOut(String sessionId) {
        SessionInternal sessionInternal = removeSession(sessionId);
        if (sessionInternal == null) {
            return;
        }
        PostMethod method = null;
        try {
            RepositoryHttpConfig repoConfig = institutionRegistry.getRepositoryHttpConfig(sessionInternal.session.getInstitutionId());
            method = new PostMethod(repoConfig.getLogoutUrl());
            int status = 0;
            if (isDebugEnabled) {
                log.debug("Trying to logout at " + repoConfig.getLogoutUrl());
            }
            status = sessionInternal.httpClient.executeMethod(method);
            // check to see if we succeeded
            if (status == HttpStatus.SC_OK) {
                if (isDebugEnabled) {
                    log.debug("logout succeded");
                }
            } else {
                log.info("Could not logout user: HTTP Status: " + status);
            }
        } catch (HttpException e) {
            log.warn("Error on logout: " + e.getMessage(), e);
        } catch (IOException e) {
            log.warn("Error on logout: " + e.getMessage(), e);
        } finally {
            if (method != null) {
                method.releaseConnection();
            }
        }
    }
    
    private SessionInternal removeSession(String sessionId) {
        synchronized(mapSessionIdToSession) {
            SessionInternal sessionInternal = mapSessionIdToSession.remove(sessionId);
            if (sessionInternal == null) {
                return null;
            }
            if (isDebugEnabled) {
                log.debug("Remove session for user: " + sessionInternal.session.getUserName() + " session id: " + sessionId);
            }
            return sessionInternal;
        }
    }
    
    private String getStringFromInputStream(InputStream in) throws IOException {
        StringWriter writer = new StringWriter();
        //encoding needs to be explicitly set
        IOUtils.copy(in, writer, "UTF-8");
        return writer.toString();
    }

    @Override
    public Session getSession(String sessionId) {
        synchronized(mapSessionIdToSession) {
            SessionInternal sessionInternal = mapSessionIdToSession.get(sessionId);
            if (sessionInternal == null) {
                return null;
            }
            // Check if the session has expired.
            long currentTime = System.currentTimeMillis();
            if (currentTime > sessionInternal.lastAccess + SESSION_TIMEOUT) {
                removeSession(sessionId);
                return null;
            }
            // If not, set last access time.
            sessionInternal.lastAccess = currentTime;
            return sessionInternal.session;
        }
    }

    @Override
    public boolean isValid(String sessionId) {
        return (getSession(sessionId) != null);
    }
}
