package org.eaglei.services.repository;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpConnectionManager;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIURI;
import org.eaglei.model.jena.MetadataConstants;
import org.eaglei.security.Session;
import org.eaglei.services.InstitutionRegistry;
import org.eaglei.services.repository.RepositoryHttpConfig.RepositoryLocale;
import org.eaglei.services.repository.RepositoryProviderException.RepositoryProviderExceptionType;

import com.hp.hpl.jena.query.QuerySolution;
import com.hp.hpl.jena.query.ResultSet;
import com.hp.hpl.jena.query.ResultSetFactory;
import com.hp.hpl.jena.rdf.model.Literal;
import com.hp.hpl.jena.rdf.model.Resource;

/**
 * 
 * @author Ricardo De Lima
 * @author Lucy Hadden
 * @author Daniela Bourges
 * @author Ted Bashor
 * 
 *         April 11, 2010
 * 
 *         Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 * 
 * 
 * 
 */
public final class RepositorySecurityProvider extends AbstractRepositoryProvider implements SecurityProvider {
    
    /** Automatic logout will occur sometime after this amount of idleness in msec, 4 hours default */
	private static final Long SESSION_TIMEOUT = Long.getLong("org.eaglei.session.timeout", 14400000);

    /*
	 * FIXME:at the moment these two workspaces are excluded when filling the workspace list in User object
	 */
	private static String WITDRAWN_WORKSPACE_URI = "http://eagle-i.org/ont/repo/1.0/NG_Withdrawn";
	private static String SANDBOX_WORKSPACE_URI = "http://eagle-i.org/ont/repo/1.0/NG_Sandbox";

	/** Logger for this class. */
    private static final Log log = LogFactory.getLog(RepositorySecurityProvider.class);
    /** Flag for debug mode */
    private static final boolean isDebugEnabled = log.isDebugEnabled();

    /** Institution registry contains network urls for various institutions. Configured by Spring. */
    private final InstitutionRegistry institutionRegistry;
    
    /**
     * Initialize from InstitutionRegistry configuration info.
     */
    public RepositorySecurityProvider(InstitutionRegistry institutionRegistry) {
        this.institutionRegistry = institutionRegistry;
        SessionReaper reaper = new SessionReaper();
        reaper.start();
        log.info("RepositorySecurityProvider initialized.  Session timeout:  " + SESSION_TIMEOUT/60000 + " min");
    }

    @Override
    public Session logIn(final String institutionId, final String user, final String password) throws RepositoryProviderException {
        HttpClient userClient = RepositoryHttpConfig.createHttpClient(user, password);
        userClient.setHttpConnectionManager(new MultiThreadedHttpConnectionManager());
       
        if (isDebugEnabled) {
            log.debug("Logging in user: " + user);
        }
        
        return getUserInformation(institutionId, null, userClient);
    }

    /** Queries the repository for user information and then encapsulates the result in a 
     * Session object which is then returned.
     * 
     * @param institutionId The ontology URI for the institution.
     * @param repoSession The RepositorySession to use and update. If it is null or invalid
     * then a new RepositorySession will be created. 
     * @param client The Http client to use for communication. May not be null.
     * @return A Session object encapsulating the information returned from the repository.
     * @throws RepositoryProviderException
     */
    private Session getUserInformation(final String institutionId, RepositorySession repoSession, final HttpClient client) throws RepositoryProviderException {
		// This method can take null session as argument,
		// in which case it creates a new session
		if ( client == null ) {
			log.error( "http Client is null" );
			throw new RepositoryProviderException( "trying to use null http client" );
		}
		// null institution id param is OK, on an institutional node
	    RepositoryHttpConfig repoConfig = institutionRegistry.getRepositoryHttpConfig(institutionId);
	    if (repoConfig == null) {
	    	String msg = "Unrecognized login institution id: " + institutionId;
	        log.error(msg);
	        throw new RepositoryProviderException(msg, RepositoryProviderExceptionType.NOT_FOUND);
	    }
	
	    if (isDebugEnabled) {
	        log.debug("Checking for user info with whoami URL of: " + repoConfig.getFullRepositoryUrl(RepositoryLocale.WHOAMI_URL));
	    }
		
	    ResultSet results = null;
	    String responseBody = null;
	    final GetMethod method = new GetMethod(repoConfig.getFullRepositoryUrl(RepositoryLocale.WHOAMI_URL));
	    try {
	        responseBody = ProviderUtils.getHttpResponse(client, method);
	        
	        if ( responseBody == null ) {
				throw new RepositoryProviderException( "got null responsebody from whoami", RepositoryProviderExceptionType.UNAUTHORIZED );
			}
	        
	        results = ResultSetFactory.fromXML(responseBody);
	       
	        if ( results == null ) {
				throw new RepositoryProviderException( "got null result set from whoami", RepositoryProviderExceptionType.UNAUTHORIZED );
			}
	        
	        // we are authorized so let's return the user name
	        final QuerySolution soln = results.nextSolution();
	        final Literal username = soln.getLiteral("username");
	        final Resource userURI = soln.getResource("uri");
	        if ( username == null || userURI == null ) {
				throw new RepositoryProviderException( "got null user/usederUri from whoami", RepositoryProviderExceptionType.UNAUTHORIZED );
			}
	        if (repoSession == null || !repoSession.isValid()) {
	        	if (repoSession != null) {
	        		SessionManager.removeSession(repoSession.getSessionId());
	        	}
		        String sessionId = UUID.randomUUID().toString();
		        Session newSession = new Session(sessionId, institutionId, username.getString(), userURI.getURI());
		        repoSession = SessionManager.addSession(client, newSession);
	        }
	        repoSession.setLastAccess(new Long(System.currentTimeMillis()));
	        if (isDebugEnabled) {
	            log.debug("Authenticated user: " + repoSession.getUserName() + " session id: " + repoSession.getSessionId());
	        }
	        return repoSession.getInnerSession();           
	    } catch (RepositoryProviderException e) {
	    	log.error(e.getMessage());
	    	throw e;
	    }
	    catch (final Exception e) {
	    	String msg = "problem getting user info " + repoConfig.getFullRepositoryUrl(RepositoryLocale.WHOAMI_URL) + " Message from repo: "
            + responseBody + "; Exception " + e;
	        log.error(msg);
	        throw new RepositoryProviderException(msg);
	    } finally {
	        method.releaseConnection();
	    }
    }
    
    @Override
    public void logOut(String sessionId) throws RepositoryProviderException {
        RepositorySession repoSession = SessionManager.getSession(sessionId);
        if (repoSession == null) {
            return;
        }
        PostMethod method = null;
        try {
            RepositoryHttpConfig repoConfig = institutionRegistry.getRepositoryHttpConfig(repoSession.getInstitutionId());
            method = new PostMethod(repoConfig.getFullRepositoryUrl(RepositoryLocale.LOGOUT_URL));
            if (isDebugEnabled) {
                log.debug("Trying to logout at " + repoConfig.getFullRepositoryUrl(RepositoryLocale.LOGOUT_URL));
            }
            ProviderUtils.getHttpResponse(repoSession.getHttpClient(), method);
            if (isDebugEnabled) {
                log.debug("logout succeded");
            }
        } catch (RepositoryProviderException e) {
            log.warn("Error on logout: " + e.getMessage(), e);
            throw e;
        } finally {
            if (method != null) {
                method.releaseConnection();
            }
            SessionManager.removeSession(sessionId);
        }
    }
    
    @Override
	public Session whoami(final String sessionId) throws RepositoryProviderException {
		if (!isValid(sessionId, false)) {
			log.info( "Using invalid session.  Could not whoami user" );
			SessionManager.removeSession(sessionId);
			return null;
		}

		RepositorySession repoSession = SessionManager.getSession(sessionId);
		
		if (repoSession == null || !repoSession.isValid()) {
			log.info( "Using invalid session.  Could not whoami user" );
			SessionManager.removeSession(sessionId);
			return null;
		}
		
		RepositoryHttpConfig repoConfig = institutionRegistry.getRepositoryHttpConfig(repoSession.getInstitutionId());
		// now let's try to do a whoami if it succeeds then we are logged in
		// otherwise return null
		if ( isDebugEnabled ) {
			log.debug( "Trying to whoami at " + repoConfig.getFullRepositoryUrl(RepositoryLocale.WHOAMI_URL) );
		}

		return getUserInformation( repoSession.getInstitutionId(), repoSession, repoSession.getHttpClient() );
	}
    
    /** Checks the validity of the passed in session object and the associated RepositorySession
	 * object. Optionally throws an exception for invalid sessions if the boolean flag
	 * shouldThrow is set to true.
	 * 
	 * @param session The Session object being validated.
	 * @param shouldThrow If true then an invalid session will result in an exception, if false
	 * then an invalid session will result in a log message and a return of false.
	 * @return True if the session and the associated RepositorySession are valid. False otherwise.
	 * @throws RepositoryProviderException If the boolean shouldThrow flag is set to true and the
	 * passed in session is invalid.
	 */
	public boolean isValid(String sessionId, boolean shouldThrow) throws RepositoryProviderException {
		if (sessionId == null) {
			if (shouldThrow) {
				throw new RepositoryProviderException("Null session id", RepositoryProviderExceptionType.INVALID_SESSION);
			}
			log.error( "Null session id" );
			return false;
		}
		if (!SessionManager.hasSession(sessionId)) {
			if (shouldThrow) {
				throw new RepositoryProviderException("Invalid Session - non-existent session id", RepositoryProviderExceptionType.INVALID_SESSION);
			}
			log.error( "Invalid Session - non-existent session id" );
			return false;
		}
		RepositorySession repoSession = SessionManager.getSession(sessionId);
		if ( repoSession == null || !repoSession.isValid() ) {
			if (shouldThrow) {
				throw new RepositoryProviderException("Session invalid or null or client null", RepositoryProviderExceptionType.INVALID_SESSION);
			}
			log.error( "Session invalid or null or client null" );
			return false;
		}
		return true;
	}
	
	/** Retrieves the HttpClient associated with the passed in Session object (if one exists).
	 * 
	 * @param session The session whose HttpClient is sought.
	 * @return The HttpClient associated with the Session object or null if none exists.
	 */
	public HttpClient getHttpClient(Session session) {
		RepositorySession repoSession = SessionManager.getSession(session.getSessionId());
		return repoSession==null?null:repoSession.getHttpClient();
	}
	
	@Override
	public boolean isOnline(String institutionId) {
		RepositoryHttpConfig repoConfig = institutionRegistry.getRepositoryHttpConfig(institutionId);
		boolean online = false;
		String url = repoConfig.getBaseURL();
		final GetMethod method = new GetMethod(url + "/admin");
		final HttpClient client = new HttpClient();
		client.setHttpConnectionManager( new MultiThreadedHttpConnectionManager() );
		if ( isDebugEnabled ) {
			log.debug( "Trying to see if Repository is available: " + url);
		}
		try {
			ProviderUtils.getHttpResponse(client, method);
			if ( isDebugEnabled ) {
				log.debug( "Repository is available: " + url + " is available with status 'OK'.");
			}
		} catch (RepositoryProviderException e) {
			if (e.getExceptionType() == RepositoryProviderExceptionType.UNAUTHORIZED) {
				online = true;
				if ( isDebugEnabled ) {
					log.debug( "Repository is available: " + url + " is available with status 'Unauthorized'.");
				}
			}
			else {
				log.error( "problem checking online status of repository: " + url + " " + e );
			}
		} finally {
			method.releaseConnection();
		}
		return online;
	}
	
	private EIEntity getEntityFromSolution(final QuerySolution solution, final String uriVariable, final String labelVariable) {
		if ( solution.contains( uriVariable ) ) {
			final EIURI uri = EIURI.create( solution.getResource( uriVariable ).getURI() );
			String label = "<none>";
			if ( solution.contains( labelVariable ) ) {
				label = solution.getLiteral( labelVariable ).getString();
			}
			return EIEntity.create( uri, label );
		} else {
			return EIEntity.NULL_ENTITY;
		}
	}
	
	@Override
	public List<HashMap<String, EIEntity>> listWorkFlowTransitions(String institutionId, final String sessionId, final EIEntity workspaceEntity) throws RepositoryProviderException {

		isValid( sessionId, true);
		RepositoryHttpConfig repoConfig = institutionRegistry.getRepositoryHttpConfig(institutionId);
		final PostMethod method = new PostMethod( repoConfig.getFullRepositoryUrl(RepositoryLocale.LIST_TRANSITIONS_URL) );
		method.setParameter( "format", FORMAT_VALUE );
		/*
		 * if not NULL ENTITY or not null then set the URI parameter otherwise don't set which defaults to list from all workspaces
		 */

		if ( workspaceEntity != null && !workspaceEntity.equals( EIEntity.NULL_ENTITY ) ) {
			method.setParameter( "URI", workspaceEntity.getURI().toString() );

		}

		log.info( "executing " + repoConfig.getFullRepositoryUrl(RepositoryLocale.LIST_TRANSITIONS_URL) + " api call" );
		HttpClient client = getHttpClient( SessionManager.getSession(sessionId).getInnerSession() );
		final String responseBody = ProviderUtils.getHttpResponse( client, method );
		// if reponseBody equals null throw Exception
		if ( responseBody == null ) {
			throw new RepositoryProviderException( "response to listWorkflowTransitions is null" );
		}
		log.info( "parsing response to make list of workflow transitions" );
		final ResultSet resultSet = ResultSetFactory.fromXML( responseBody );
		final List<HashMap<String, EIEntity>> transitionList = new ArrayList<HashMap<String, EIEntity>>();
		while ( resultSet.hasNext() ) {
			final QuerySolution solution = resultSet.next();
			// Only process solution if the contained transition is allowed
			if ( solution.contains( WORKFLOW_TRANSITION_ALLOWED ) ) {
				final boolean allowed = solution.getLiteral( WORKFLOW_TRANSITION_ALLOWED ).getBoolean();
				if ( allowed ) {
					HashMap<String, EIEntity> workFlowTransition = new HashMap<String, EIEntity>(3);
					workFlowTransition.put(WORKFLOW_TRANSITION_SUBJECT, getEntityFromSolution( solution, WORKFLOW_TRANSITION_SUBJECT, WORKFLOW_TRANSITION_LABEL ));
					workFlowTransition.put(WORKFLOW_TRANSITION_FROM, getEntityFromSolution( solution, WORKFLOW_TRANSITION_FROM, WORKFLOW_TRANSITION_FROM_LABEL ));
					workFlowTransition.put(WORKFLOW_TRANSITION_TO, getEntityFromSolution( solution, WORKFLOW_TRANSITION_TO, WORKFLOW_TRANSITION_TO_LABEL ));
					transitionList.add( workFlowTransition );
				}
			}
		}
		//TODO: Throw if list size is zero (unauthorized user)
		log.info( transitionList.size() + " transitions were returned" );
		return transitionList;
	}
	
	@Override
	public List<HashMap<String, String>> getWorkspaces(String institutionId, String sessionId) throws RepositoryProviderException {
		isValid( sessionId, true);

		RepositoryHttpConfig repoConfig = institutionRegistry.getRepositoryHttpConfig(institutionId);
		String methodURL = repoConfig.getFullRepositoryUrl(RepositoryLocale.LIST_GRAPHS_URL);
		final PostMethod method = new PostMethod( methodURL );
		method.setParameter( "format", FORMAT_VALUE );
		// FIXME we need to figure out how to restrict this
		// method.setParameter( "type", "workspace" );
		log.info( "executing " + methodURL + " api call" );

		RepositorySession repoSession = SessionManager.getSession(sessionId);
		final String responseBody = ProviderUtils.getHttpResponse( repoSession.getHttpClient(), method );
		log.info( methodURL + " api call has returned ,parsing the results" );
		final List<HashMap<String, String>> workspaceList = new ArrayList<HashMap<String, String>>();
		final ResultSet resultSet = ResultSetFactory.fromXML( responseBody );
		while ( resultSet.hasNext() ) {
			final QuerySolution solution = resultSet.next();
			final String workspaceURI = solution.getResource( MetadataConstants.WORKSPACE_NAMED_GRAPH_URI ).getURI();
			// FIXME:at the moment hard code to exclude two workspaces
			if ( workspaceURI.equals( WITDRAWN_WORKSPACE_URI ) || workspaceURI.equals( SANDBOX_WORKSPACE_URI ) ) {
				continue;
			}
			HashMap<String, String> workspaceMap = new HashMap<String, String>(5);
			workspaceMap.put(MetadataConstants.WORKSPACE_NAMED_GRAPH_LABEL, solution.getLiteral( MetadataConstants.WORKSPACE_NAMED_GRAPH_LABEL ).getString());
			workspaceMap.put(MetadataConstants.WORKSPACE_NAMED_GRAPH_URI, solution.getResource( MetadataConstants.WORKSPACE_NAMED_GRAPH_URI ).getURI());
			workspaceMap.put(MetadataConstants.WORKSPACE_TYPE, solution.getResource( MetadataConstants.WORKSPACE_TYPE ).getURI());
			workspaceMap.put(MetadataConstants.WORKSPACE_ADD, String.valueOf(solution.getLiteral( MetadataConstants.WORKSPACE_ADD ).getBoolean()));
			workspaceMap.put(MetadataConstants.WORKSPACE_REMOVE, String.valueOf(solution.getLiteral( MetadataConstants.WORKSPACE_REMOVE ).getBoolean()));
			workspaceList.add( workspaceMap );

		}
		return workspaceList;
	}
	
	@Override
	public Session getSession(String sessionId) {
		RepositorySession session = SessionManager.getSession(sessionId);
		return session!=null?session.getInnerSession():null;
	}
	
	/** Utility class for handling session operations. All calls are synchronized.
	 * 
	 * 
	 */
	private static class SessionManager {
		/** HashMap that maps session IDs to the repository session object. */
		private static Map<String, RepositorySession> mapSessionIdToRepoSession = new HashMap<String, RepositorySession>();

		/** Retrieves the RepositorySession object with the specified session id
		 * from the session id map if such a session exists. Otherwise returns null.
		 * 
		 * @param sessionId The session id of the desired repository session.
		 * @return A RepositorySession if one exists in the map with the specified id, 
		 * null otherwise.
		 */
		public static RepositorySession getSession(String sessionId) {
	        synchronized(mapSessionIdToRepoSession) {
	            RepositorySession repoSession = mapSessionIdToRepoSession.get(sessionId);
	            if (repoSession == null) {
	                return null;
	            }
	            return repoSession;
	        }
	    }
		
		/** Removes the RepositorySession from the session map with the passed
		 * in session id (if one exists) and returns it.
		 * 
		 * @param sessionId The session id of the desired repository session.
		 * @return The RepositorySession with the specified id, null if no
		 * RepositorySession in the map has the id.
		 */
		public static RepositorySession removeSession(String sessionId) {
	        synchronized(mapSessionIdToRepoSession) {
	        	RepositorySession repoSession = mapSessionIdToRepoSession.remove(sessionId);
	            if (repoSession == null) {
	                return null;
	            }
		    // We've purged. Close all connections immediately.
		    HttpConnectionManager connectionManager = repoSession.getHttpClient().getHttpConnectionManager();
		    if ( connectionManager instanceof MultiThreadedHttpConnectionManager ) {
			((MultiThreadedHttpConnectionManager)connectionManager).shutdown();
		    } else {
			repoSession.getHttpClient().getHttpConnectionManager().closeIdleConnections(0);
		    }
		    if (isDebugEnabled) {
	                log.debug("Remove session for user: " + repoSession.getUserName() + " session id: " + sessionId);
	            }
	            return repoSession;
	        }
	    }
		
		/** Checks the session map to see if it contains a RepositorySession with the
		 * specified session id.
		 * 
		 * @param sessionId The session id of the RepositorySession to check for.
		 * @return True if the session map contains a RepositorySession with the
		 * specified session id. False otherwise.
		 */
		public static boolean hasSession(String sessionId) {
			synchronized(mapSessionIdToRepoSession) {
				return mapSessionIdToRepoSession.containsKey(sessionId);
			}
		}
		
		/** Creates a new RepositorySession using the passed in HttpClient and Session.
		 * This RepositorySession is then added to the session map and returned.
		 * 
		 * @param client The HttpClient to use in creating the new RepositorySession.
		 * @param session The Session to use in creating the new RepositorySession.
		 * @return The new RepositorySession.
		 * @throws RepositoryProviderException If the passed in client is null or the
		 * passed in Session is invalid.
		 */
		public static RepositorySession addSession(HttpClient client, Session session) throws RepositoryProviderException {
			RepositorySession newSession = new RepositorySession(session, client);
			if (!newSession.isValid()) {
				throw new RepositoryProviderException("Tried to add repository session with invalid session or null client.");
			}
			synchronized(mapSessionIdToRepoSession) {
				mapSessionIdToRepoSession.put(newSession.getSessionId(), newSession);
			}
			return newSession;
		}
		
		/** Retrieves the set of session ids currently in the session id map.
		 * 
		 * @return A Set of session ids.
		 */
		public static Set<String> getSessionIdSet() {
			synchronized(mapSessionIdToRepoSession) {
			    return new HashSet<String>( mapSessionIdToRepoSession.keySet() );
			}
		}

	}
	
	/** Class for removing expired sessions from the session map.
	 * 
	 * 
	 *
	 */
	private class SessionReaper extends Thread {
		
		/** The interval at which the thread checks sessions for expiration. */
		private final long SLEEP_INTERVAL = SESSION_TIMEOUT / 8;
        
        SessionReaper() {
            setDaemon(true);
        }
        
        /** Starts the thread and enters a continuous loop in which the
         * session map is checked for expired sessions at an interval
         * specified by a constant.
         * 
         */
        @Override
        public void run() {
            while (true) {
                try {
                    Thread.sleep(SLEEP_INTERVAL);
                } catch (InterruptedException e) {
		    log.warn("SessionReaper interrupted", e);
		    SessionReaper newReaper = new SessionReaper();
		    newReaper.start();
                    return;
                }
                Set<String> sessionList = SessionManager.getSessionIdSet();

                if (log.isDebugEnabled()) {
		    log.debug("Starting to reap (list currently contains " + sessionList.size() + " sessions)");
                }
                
                for (String sessionId : sessionList) {
                	RepositorySession repoSession = SessionManager.getSession(sessionId);
			if (! purgeExpiredSession(repoSession)) {
			    // Check for anything idle for awhile (>=30mins).
			    repoSession.getHttpClient(false).getHttpConnectionManager().closeIdleConnections(SLEEP_INTERVAL);
			}
		}
	    }
	}
        
        /** Checks the passed in RepositorySession to see if it is expired.
         * If it is then it is removed from the session map.
         * 
         * @param repoSession The RepositorySession being checked for expiration.
         * @return True if the session was expired and removed, false otherwise.
         */
        private boolean purgeExpiredSession(RepositorySession repoSession) {
        	// Check if the session has expired.
            long currentTime = System.currentTimeMillis();
            if (!repoSession.isValid() || currentTime > repoSession.getLastAccess() + SESSION_TIMEOUT) {
                SessionManager.removeSession(repoSession.getSessionId());
                return true;
            }
            return false;
        }
    }
}
